<?php

namespace Asics\Verifone\Cron;

use Exception;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Stdlib\DateTime\DateTime;
use Magento\Sales\Api\Data\OrderInterface;
use Magento\Sales\Model\ResourceModel\Order as OrderResource;
use Psr\Log\LoggerInterface;
use Magento\Sales\Model\Order;
use Throwable;
use Verifone\Hosted\Model\Verifone;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Model\Order\Status\HistoryFactory;

class CreateInvoice
{
    /**
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param DateTime $dateTime
     * @param OrderRepositoryInterface $orderRepository
     * @param HistoryFactory $orderHistoryFactory
     * @param Verifone $verifone
     * @param LoggerInterface $logger
     */
    public function __construct(
        private readonly SearchCriteriaBuilder $searchCriteriaBuilder,
        private readonly DateTime $dateTime,
        private readonly OrderRepositoryInterface $orderRepository,
        private readonly OrderResource $orderResource,
        private readonly HistoryFactory $orderHistoryFactory,
        private readonly Verifone $verifone,
        private readonly LoggerInterface $logger
    ) {
    }

    /**
     * @return void
     */
    public function execute()
    {
        $defaultStatus = $this->getDefaultVerifoneOrderStatus();
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter('extension_attribute_payment_method_code.method', Verifone::CODE)
            ->addFilter(OrderInterface::STATUS, $defaultStatus, 'in')
            ->addFilter(OrderInterface::CREATED_AT, $this->dateTime->gmtDate(input: '-1 day'), 'gteq')
            ->create();

        /** @var \Magento\Sales\Model\Order[] $orders */
        $orders = $this->orderRepository->getList($searchCriteria)->getItems();

        foreach ($orders as $order) {
            try {
                if (!$order->hasInvoices()) {
                    $payment = $order->getPayment()->getAdditionalInformation();

                    if (isset($payment['verifone_order_status'], $payment['verifone_order_number'])
                        && $payment['verifone_order_status'] === 'approved'
                        && $payment['verifone_order_number']
                    ) {
                        $this->_markOrderAsPaid($order, $payment['verifone_order_number']);
                    }
                }
            } catch (Throwable $exception) {
                $this->logger->error(
                    'Failed to create invoice for Verifone order',
                    ['order_id' => $order->getId(), 'exception' => $exception]
                );
            }
        }
    }

    /**
     * @param Order $order
     * @param $transactionId
     * @return void
     * @throws LocalizedException
     */
    public function _markOrderAsPaid(Order $order, $transactionId)
    {
        $amount = $order->getGrandTotal();
        $this->logger->info(
            'Create Invoice for Verifone Order',
            ['order_id' => $order->getEntityId(), 'transaction_id' => $transactionId, 'amount' => $amount]
        );
        if ($order->getEntityId()) {
            $this->_invoice($order, $transactionId);
            $message = __(
                'Invoiced amount of %1 Transaction ID: %2',
                $order->formatPriceTxt($amount),
                $transactionId
            );
            $this->_addHistoryComment($order, $message);
        }
    }

    /**
     * @param Order $order
     * @param $transactionId
     * @return void
     * @throws LocalizedException
     */
    public function _invoice(Order $order, $transactionId)
    {
        $invoice = $order->prepareInvoice();
        $invoice->getOrder()->setIsInProcess(true);

        $order->setState(Order::STATE_PROCESSING);
        $order->setStatus(Order::STATE_PROCESSING);
        $order->setSendEmail(true);

        $invoice->setTransactionId($transactionId);
        $invoice->register()
            ->pay()
            ->save();

        $this->orderResource->saveAttribute($order, [OrderInterface::STATE, OrderInterface::STATUS, 'send_email']);
    }

    /**
     * Adds a comment to order history
     *
     * @param Order $order
     * @param string $message
     * @throws Exception
     */
    public function _addHistoryComment($order, $message)
    {
        $history = $this->orderHistoryFactory->create()
          ->setComment($message)
          ->setEntityName('order')
          ->setOrder($order);

        $history->save();
    }

    /**
     * @return mixed|string
     */
    public function getDefaultVerifoneOrderStatus()
    {
        if (isset($this->verifone->getConfigData('account_settings')['order_status'])
            && $this->verifone->getConfigData('account_settings')['order_status']) {
            return $this->verifone->getConfigData('account_settings')['order_status'];
        }
        return 'pending, processing';
    }
}
