<?php
declare(strict_types=1);

namespace Overdose\Checkout\ViewModel;

use Magento\Framework\View\Element\Block\ArgumentInterface;
use Magento\Msrp\Helper\Data as MsrpHelper;
use Magento\Framework\Pricing\Helper\Data as PricingHelper;
use BrittainWynyard\CheckoutAttributes\Helper\Data as CheckoutAttributesHelper;
use Overdose\Catalog\Helper\Stock as StockHelper;

class CheckoutHelpersViewModel implements ArgumentInterface
{
    /**
     * @param MsrpHelper $msrpHelper
     * @param PricingHelper $pricingHelper
     * @param CheckoutAttributesHelper $checkoutAttributesHelper
     * @param StockHelper $stockHelper
     */
    public function __construct(
        private readonly MsrpHelper $msrpHelper,
        private readonly PricingHelper $pricingHelper,
        private readonly CheckoutAttributesHelper $checkoutAttributesHelper,
        private readonly StockHelper $stockHelper
    ) {}

    /**
     * Get MSRP helper
     *
     * @return MsrpHelper
     */
    public function getMsrpHelper(): MsrpHelper
    {
        return $this->msrpHelper;
    }

    /**
     * Get pricing helper
     *
     * @return PricingHelper
     */
    public function getPricingHelper(): PricingHelper
    {
        return $this->pricingHelper;
    }

    /**
     * Get checkout attributes helper
     *
     * @return CheckoutAttributesHelper
     */
    public function getCheckoutAttributesHelper(): CheckoutAttributesHelper
    {
        return $this->checkoutAttributesHelper;
    }

    /**
     * Get stock helper
     *
     * @return StockHelper
     */
    public function getStockHelper(): StockHelper
    {
        return $this->stockHelper;
    }

    /**
     * Check if MSRP price must be shown before order confirmation
     *
     * @param \Magento\Catalog\Model\Product $product
     * @return bool
     */
    public function isShowBeforeOrderConfirm($product)
    {
        return $this->getMsrpHelper()->isShowBeforeOrderConfirm($product);
    }

    /**
     * Check if minimal price is less than MSRP
     *
     * @param \Magento\Catalog\Model\Product $product
     * @return bool
     */
    public function isMinimalPriceLessMsrp($product)
    {
        return $this->getMsrpHelper()->isMinimalPriceLessMsrp($product);
    }

    /**
     * Get product attribute id by attribute code
     *
     * @param string $code
     * @return int|null
     */
    public function getProductAttributeIdByCode($code)
    {
        return $this->getCheckoutAttributesHelper()->getProductAttributeIdByCode($code);
    }

    /**
     * Get product attribute value by attribute code
     *
     * @param \Magento\Catalog\Model\Product $product
     * @param string $code
     * @return mixed
     */
    public function getProductAttributeValueByCode($product, $code) {
        return $this->getCheckoutAttributesHelper()->getProductAttributeValueByCode($product, $code);
    }

    /**
     * Format currency value
     *
     * @param float|int|string $value
     * @param bool $format
     * @param bool $includeContainer
     * @return string
     */
    public function getCurrency($value, $format = true, $includeContainer = true)
    {
        return $this->getPricingHelper()->currency($value, $format, $includeContainer);
    }
}