<?php

namespace Overdose\Customer\Block\Sales;

use Magento\Catalog\Block\Product\Context;
use Magento\Catalog\Block\Product\ImageBuilder;
use Magento\Catalog\Model\Product;
use Magento\Catalog\Model\ResourceModel\Product\Collection as ProductCollection;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory as ProductCollectionFactory;
use Magento\Catalog\Pricing\Price\FinalPrice;
use Magento\Checkout\Helper\Cart;
use Magento\Customer\Model\Session;
use Magento\Framework\App\ActionInterface;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Pricing\Render;
use Magento\Framework\Url\Helper\Data;
use Magento\Sales\Api\Data\OrderInterface;
use Magento\Sales\Api\Data\OrderItemInterface;
use Magento\Sales\Model\Order\Config;
use Magento\Sales\Model\ResourceModel\Order\Collection as OrderCollection;
use Magento\Sales\Model\ResourceModel\Order\CollectionFactory as OrderCollectionFactory;
use Magento\Store\Model\StoreManagerInterface;

/**
 * Sales reorder products block
 */
class Reorder extends \Magento\Framework\View\Element\Template
{
    /**
     * @var OrderCollectionFactory
     */
    protected $_orderCollectionFactory;

    /**
     * @var Session
     */
    protected $_customerSession;

    /**
     * @var Config
     */
    protected $_orderConfig;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var ProductCollectionFactory
     */
    private $_productCollection;

    /**
     * @var Cart
     */
    protected $_cartHelper;

    /**
     * @var ImageBuilder
     */
    protected $imageBuilder;

    /**
     * @var Data
     */
    protected $urlHelper;
    /**
     * @param Context $context
     * @param OrderCollectionFactory $orderCollectionFactory
     * @param ProductCollectionFactory $productCollectionFactory,
     * @param Session $customerSession
     * @param Config $orderConfig
     * @param Data $urlHelper
     * @param StoreManagerInterface $storeManager
     * @param array $data
     */
    public function __construct(
        Context $context,
        OrderCollectionFactory $orderCollectionFactory,
        Session $customerSession,
        Config $orderConfig,
        ProductCollectionFactory $productCollectionFactory,
        Data $urlHelper,
        StoreManagerInterface $storeManager = null,
        array $data = []
    ) {
        $this->_orderCollectionFactory = $orderCollectionFactory;
        $this->_customerSession = $customerSession;
        $this->_orderConfig = $orderConfig;
        $this->_productCollection = $productCollectionFactory;
        $this->imageBuilder = $context->getImageBuilder();
        $this->_cartHelper = $context->getCartHelper();
        $this->urlHelper = $urlHelper;
        $this->storeManager = $storeManager ?: ObjectManager::getInstance()
            ->get(StoreManagerInterface::class);
        parent::__construct($context, $data);
    }

    /**
     * @inheritDoc
     */
    protected function _construct()
    {
        parent::_construct();
    }

    /**
     * Get ordered products
     */
    public function getOrderedProducts()
    {
        $orders = $this->getCustomerOrders();
        $productIds = $this->getOrderedProductIds($orders);
        return $this->getOrderedProductsCollection($productIds);
    }

    /**
     * @return OrderCollection
     */
    private function getCustomerOrders()
    {
        return $this->_orderCollectionFactory->create()->addAttributeToSelect(
            '*'
        )->addAttributeToFilter(
            'customer_id',
            $this->_customerSession->getCustomerId()
        )->addAttributeToFilter(
            'main_table.store_id',
            $this->storeManager->getStore()->getId()
        )->addAttributeToFilter(
            'main_table.status',
            ['in' => $this->_orderConfig->getVisibleOnFrontStatuses()]
        )->addAttributeToSort(
            'main_table.created_at',
            'desc'
        );
    }

    /**
     * Sort ordered products by quantity purchased
     * @param OrderCollection $orders
     * @return array
     */
    private function getOrderedProductIds($orders)
    {
        $productIds = [];
        /** @var OrderInterface $order */
        foreach ($orders as $order) {
            $items = $order->getAllVisibleItems();
            /** @var $item OrderItemInterface */
            foreach ($items as $item) {
                if (!array_key_exists($item->getProductId(), $productIds)) {
                    $productIds[$item->getProductId()] = 1;
                } else {
                    $productIds[$item->getProductId()]++;
                }
            }
        }
        arsort($productIds);
        return array_keys($productIds);
    }

    /**
     * @param $ids
     * @return ProductCollection
     */
    private function getOrderedProductsCollection($ids)
    {
        $productCollection = $this->_productCollection->create()
            ->addAttributeToFilter('entity_id', ['in' => implode(",", $ids)])
            ->addAttributeToSelect('*');
        if (count($ids)) {
            $productCollection->getSelect()->order(new \Zend_Db_Expr('FIELD(e.entity_id, ' . implode(',', $ids) . ')'));
            $productCollection->getSelect()->limit($this->numProductsToShow());
        }
        return $productCollection;
    }

    /**
     * Retrieve product image
     *
     * @param \Magento\Catalog\Model\Product $product
     * @param string $imageId
     * @param array $attributes
     * @return \Magento\Catalog\Block\Product\Image
     */
    public function getImage($product, $imageId, $attributes = [])
    {
        return $this->imageBuilder->create($product, $imageId, $attributes);
    }

    /**
     * Retrieve url for add product to cart
     *
     * Will return product view page URL if product has required options
     *
     * @param \Magento\Catalog\Model\Product $product
     * @param array $additional
     * @return string
     */
    public function getAddToCartUrl($product, $additional = [])
    {
        if (!$product->getTypeInstance()->isPossibleBuyFromList($product)) {
            if (!isset($additional['_escape'])) {
                $additional['_escape'] = true;
            }
            if (!isset($additional['_query'])) {
                $additional['_query'] = [];
            }
            $additional['_query']['options'] = 'cart';

            return $this->getProductUrl($product, $additional);
        }
        return $this->_cartHelper->getAddUrl($product, $additional);
    }

    /**
     * Retrieve Product URL using UrlDataObject
     *
     * @param \Magento\Catalog\Model\Product $product
     * @param array $additional the route params
     * @return string
     */
    public function getProductUrl($product, $additional = [])
    {
        if ($this->hasProductUrl($product)) {
            if (!isset($additional['_escape'])) {
                $additional['_escape'] = true;
            }
            return $product->getUrlModel()->getUrl($product, $additional);
        }

        return '#';
    }

    /**
     * Get post parameters
     *
     * @param Product $product
     * @return array
     */
    public function getAddToCartPostParams(Product $product)
    {
        $url = $this->getAddToCartUrl($product, ['_escape' => false]);
        return [
            'action' => $url,
            'data' => [
                'product' => (int) $product->getEntityId(),
                ActionInterface::PARAM_NAME_URL_ENCODED => $this->urlHelper->getEncodedUrl($url),
            ]
        ];
    }

    /**
     * Get product price.
     *
     * @param Product $product
     * @return string
     */
    public function getProductPrice(Product $product)
    {
        $priceRender = $this->getPriceRender();
        $price = '';
        if ($priceRender) {
            $price = $priceRender->render(
                FinalPrice::PRICE_CODE,
                $product,
                [
                    'include_container' => true,
                    'display_minimal_price' => true,
                    'zone' => Render::ZONE_ITEM_LIST,
                    'list_category_page' => true
                ]
            );
        }
        return $price;
    }

    /**
     * Specifies that price rendering should be done for the list of products.
     * (rendering happens in the scope of product list, but not single product)
     *
     * @return Render
     */
    protected function getPriceRender()
    {
        return $this->getLayout()->getBlock('product.price.render.default')
            ->setData('is_product_list', true);
    }

    /**
     * Whether redirect to cart enabled
     *
     * @return bool
     */
    public function isRedirectToCartEnabled()
    {
        return $this->_scopeConfig->getValue(
            'checkout/cart/redirect_to_cart',
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE
        );
    }

    public function numProductsToShow()
    {
        return $this->_scopeConfig->getValue(
            'overdose_general/configuration/customer_reorder_products_display',
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE
        );
    }
}
