<?php
namespace Overdose\Customer\Model;

use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Customer\Helper\View as CustomerViewHelper;
use Magento\Customer\Model\AccountManagement;
use Magento\Customer\Model\AddressRegistry;
use Magento\Customer\Model\CustomerRegistry;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Exception\MailException;
use Magento\Framework\Mail\Template\TransportBuilder;
use Magento\Framework\Math\Random;
use Magento\Framework\Reflection\DataObjectProcessor;
use Magento\Framework\Registry;
use Magento\Store\Model\StoreManagerInterface;
use Psr\Log\LoggerInterface as PsrLogger;
use Magento\Store\Model\ScopeInterface;

/**
 * Class SendCreatePassword
 * @package Overdose\Customer\Model
 */
class SendCreatePassword
{
    /**
     * Sender
     */
    const EMAIL_SENDER = 'customer/password/forgot_email_identity';

    /**
     * Tempate
     */
    const EMAIL_TEMPLATE = 'overdose_general/configuration/create_password_email_template';

    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var Random
     */
    private $mathRandom;

    /**
     * @var PsrLogger
     */
    protected $logger;

    /**
     * @var CustomerRegistry
     */
    private $customerRegistry;

    /**
     * @var CustomerRepositoryInterface
     */
    private $customerRepository;

    /**
     * @var ScopeConfigInterface
     */
    private $scopeConfig;

    /**
     * @var TransportBuilder
     */
    private $transportBuilder;

    /**
     * @var DataObjectProcessor
     */
    protected $dataProcessor;

    /**
     * @var \Magento\Framework\Registry
     */
    protected $registry;

    /**
     * @var CustomerViewHelper
     */
    protected $customerViewHelper;

    /**
     * @var AddressRegistry
     */
    private $addressRegistry;

    /**
     * @var AccountManagement
     */
    private $accountManagement;

    /**
     * SendCreatePassword constructor.
     * @param StoreManagerInterface $storeManager
     * @param Random $mathRandom
     * @param CustomerRegistry $customerRegistry
     * @param PsrLogger $logger
     * @param CustomerRepositoryInterface $customerRepository
     * @param ScopeConfigInterface $scopeConfig
     * @param TransportBuilder $transportBuilder
     * @param DataObjectProcessor $dataProcessor
     * @param Registry $registry
     * @param CustomerViewHelper $customerViewHelper
     * @param AddressRegistry $addressRegistry
     * @param AccountManagement $accountManagement
     */
    public function __construct(
        StoreManagerInterface $storeManager,
        Random $mathRandom,
        CustomerRegistry $customerRegistry,
        PsrLogger $logger,
        CustomerRepositoryInterface $customerRepository,
        ScopeConfigInterface $scopeConfig,
        TransportBuilder $transportBuilder,
        DataObjectProcessor $dataProcessor,
        Registry $registry,
        CustomerViewHelper $customerViewHelper,
        AddressRegistry $addressRegistry,
        AccountManagement $accountManagement
    ) {
        $this->storeManager = $storeManager;
        $this->mathRandom = $mathRandom;
        $this->customerRegistry = $customerRegistry;
        $this->logger = $logger;
        $this->customerRepository = $customerRepository;
        $this->scopeConfig = $scopeConfig;
        $this->transportBuilder = $transportBuilder;
        $this->dataProcessor = $dataProcessor;
        $this->registry = $registry;
        $this->customerViewHelper = $customerViewHelper;
        $this->addressRegistry = $addressRegistry;
        $this->accountManagement = $accountManagement;
    }

    /**
     * @param $email
     * @return bool
     * @throws \Magento\Framework\Exception\LocalizedException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function initiatePasswordCreate($email)
    {
        try {
            $websiteId = $this->storeManager->getStore()->getWebsiteId();

            // load customer by email
            $customer = $this->customerRepository->get($email, $websiteId);

            // No need to validate customer address while saving customer reset password token
            $this->disableAddressValidation($customer);

            $newPasswordToken = $this->mathRandom->getUniqueHash();
            $this->accountManagement->changeResetPasswordLinkToken($customer, $newPasswordToken);

            $customerEmailData = $this->getFullCustomerObject($customer);

            $storeId = $this->storeManager->getStore()->getId();

            $templateId = $this->getEmailTemplate();

            $templateParams = ['customer' => $customerEmailData->getData(), 'store' => $this->storeManager->getStore()];
            $this->transportBuilder->setTemplateIdentifier(
                $templateId
            )->setTemplateOptions(
                [
                    'area' => \Magento\Framework\App\Area::AREA_FRONTEND,
                    'store' => $storeId,
                ]
            )->setTemplateVars(
                $templateParams
            )->setFrom(
                $this->scopeConfig->getValue(
                    self::EMAIL_SENDER,
                    \Magento\Store\Model\ScopeInterface::SCOPE_STORE,
                    $storeId
                )
            )->addTo(
                $email,
                $this->customerViewHelper->getCustomerName($customer)
            );
            $transport = $this->transportBuilder->getTransport();
            $transport->sendMessage();
            return true;
        } catch (MailException $e) {
            // If we are not able to send a reset password email, this should be ignored
            $this->logger->critical($e);
            return false;
        }
    }

    /**
     * Disable Customer Address Validation
     *
     * @param CustomerInterface $customer
     * @throws NoSuchEntityException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    private function disableAddressValidation($customer)
    {
        foreach ($customer->getAddresses() as $address) {
            $addressModel = $this->addressRegistry->retrieve($address->getId());
            $addressModel->setShouldIgnoreValidation(true);
        }
    }

    /**
     * @param \Magento\Customer\Api\Data\CustomerInterface $customer
     * @return \Magento\Customer\Model\Data\CustomerSecure
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    protected function getFullCustomerObject($customer)
    {
        // No need to flatten the custom attributes or nested objects since the only usage is for email templates and
        // object passed for events
        $mergedCustomerData = $this->customerRegistry->retrieveSecureData($customer->getId());
        $customerData = $this->dataProcessor->buildOutputDataArray(
            $customer,
            \Magento\Customer\Api\Data\CustomerInterface::class
        );
        $mergedCustomerData->addData($customerData);
        $mergedCustomerData->setData('name', $this->customerViewHelper->getCustomerName($customer));
        return $mergedCustomerData;
    }

    /**
     * @return string
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getEmailTemplate()
    {
        return (string) $this->getValue(self::EMAIL_TEMPLATE);
    }

    /**
     * @param $path
     * @return mixed
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getValue($path)
    {
        $value = $this->scopeConfig->getValue(
            $path,
            ScopeInterface::SCOPE_STORE,
            $this->storeManager->getStore()->getId()
        );

        return $value;
    }
}
