<?php
declare(strict_types=1);

namespace Overdose\Seo\Plugin;

use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Catalog\Helper\Data;
use Magento\Catalog\Model\Category;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Registry;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;
use Mirasvit\Seo\Api\Service\TemplateEngineServiceInterface;
use Mirasvit\SeoMarkup\Model\Config\ProductConfig;
use Mirasvit\SeoMarkup\Service\ProductRichSnippetsService;
use Mirasvit\Seo\Helper\Data as SeoHelper;
use Magento\Catalog\Model\CategoryFactory;

class ModifyProductRichSnippets
{
    /**
     * @var int|null 
     */
    private $storeId;
    
    /**
     * @var Category
     */
    private $category;

    /**
     * @param ProductConfig $productConfig
     * @param StoreManagerInterface $storeManager
     * @param TemplateEngineServiceInterface $templateEngineService
     * @param CategoryFactory $categoryFactory
     * @param SeoHelper $seoHelper
     * @param Registry $registry
     * @param Data $catalogHelperData
     */
    public function __construct(
        private readonly ProductConfig $productConfig,
        private readonly StoreManagerInterface $storeManager,
        private readonly TemplateEngineServiceInterface $templateEngineService,
        private readonly CategoryFactory $categoryFactory,
        private readonly SeoHelper $seoHelper,
        private readonly Registry $registry,
        private readonly Data $catalogHelperData
    ) {
    }
    /**
     * @param ProductRichSnippetsService $subject
     * @param array $result
     * @param ProductInterface|null $product
     * @param bool $dry
     * @return array
     */
    public function afterGetJsonData(
        ProductRichSnippetsService $subject,
        array $result,
        ?ProductInterface $product,
        bool $dry = false
    ): array {
        if ($product) {
            $storeId = $this->getStoreId();
            $isProductGroup = $this->productConfig->isProductVariantsEnabled($storeId)
                && ($product->getTypeId() === Configurable::TYPE_CODE);
            if (!$dry) {
                if ($isProductGroup) {
                    if (isset($result['offers'])) {
                        unset($result['offers']);
                    }
                    $width = $product->getResource()->getAttribute('width')->getFrontend()->getValue($product);
                    if ($width) {
                        $result['width'] = $width;
                    }
                    $category = $this->getCategory($product, $storeId);
                    if ($category && $category->getId()) {
                        $result['category'] = [
                            '@type' => 'Thing',
                            'name' => $category->getName(),
                            'url' => $category->getUrl()
                        ];
                    }
                    if (!empty($result['hasVariant'])) {
                        foreach ($result['hasVariant'] as &$variant) {
                            $variant['offers']['hasMerchantReturnPolicy'] = $this->getReturnPolicy();
                            $variant['offers']['shippingDetails'] = $this->getShippingDetails();
                        }
                    }
                }
            } else {
                $result['size'] = $this->getSize($product);
            }
        }
        
        return $result;
    }
    /**
     * @return int
     */
    private function getStoreId(): int
    {
        if (!isset($this->storeId)) {
            try {
                $this->storeId = (int)$this->storeManager->getStore()->getId();
            } catch (NoSuchEntityException $exception) {
                return Store::DEFAULT_STORE_ID;
            }
        }
        return $this->storeId;
    }
    private function getShippingDetails(): array
    {
        return [
            '@type' => 'OfferShippingDetails',
            'shippingRate' => [
                '@type' => 'MonetaryAmount',
                'value' => 0,
                'currency' => 'NZD'
            ],
            'shippingDestination' => [
                '@type' => 'DefinedRegion',
                'addressCountry' => 'NZ'
            ],
            'deliveryTime' => [
                '@type' => 'ShippingDeliveryTime',
                'handlingTime' => [
                    '@type' => 'QuantitativeValue',
                    'minValue' => 0,
                    'maxValue' => 1,
                    'unitCode' => 'DAY'
                ],
                'transitTime' => [
                    '@type' => 'QuantitativeValue',
                    'minValue' => 2,
                    'maxValue' => 2,
                    'unitCode' => 'DAY'
                ]
            ]
        ];
    }
    private function getReturnPolicy(): array
    {
        return [
            '@type' => 'MerchantReturnPolicy',
            'applicableCountry' => 'NZ',
            'returnPolicyCategory' => 'https://schema.org/MerchantReturnFiniteReturnWindow',
            'merchantReturnDays' => 30,
            'returnFees' => 'https://schema.org/FreeReturn',
            'returnMethod' => [
                "https://schema.org/ReturnByMail",
                "https://schema.org/ReturnInStore"
            ]            
        ];
    }
    private function getSize(ProductInterface $product)
    {
        if ($attribute = $this->productConfig->getSizeAttribute($this->getStoreId())) {
            return $this->templateEngineService->render("[product_$attribute]", ['product' => $product]) ?? '';
        }
        return false;
    }
    
    private function getCategory($product, $storeId)
    {
        if ($category = $this->getCurrentCategory()) {
             return $category;
        }
        
        $breadcrumbs = $this->catalogHelperData->getBreadcrumbPath();
        $categoryId = $this->seoHelper->getPreviousCategoryId();
        if (!empty($breadcrumbs)) {
            $previousCatId = null;
            foreach ($breadcrumbs as $key => $breadcrumb) {
                if (isset($breadcrumb['link']) && $breadcrumb['link'] && str_contains($key, 'category')) {
                    $previousCatId = str_replace('category', '', $key);
                }                
            }
            $categoryId = (int) $previousCatId;
        }

        if ($categoryId) {
            $categoryIds = [$categoryId];
        } else {
            $categoryIds = $product->getCategoryIds();
            $categoryIds = array_reverse($categoryIds);
        }

        if (isset($categoryIds[0])) {
            return $this->categoryFactory->create()
                ->setStoreId($storeId)
                ->load($categoryIds[0]);
        }
    }
    
    private function getCurrentCategory()
    {
        if (!$this->category) {
            return $this->registry->registry('current_category') ?: null;
        }

        return $this->category;
    }
}