<?php
/**
 * BrittainWynyard GiftcardAccount Accredo Processor
 *
 * @package BrittainWynyard_GiftcardAccount
 */

namespace BrittainWynyard\GiftcardAccount\Model\Accredo;

use BrittainWynyard\GiftcardAccount\Api\AccredoProcessorInterface;
use BrittainWynyard\GiftcardAccount\Helper\Config;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\HTTP\Client\Curl;
use Magento\Framework\Serialize\SerializerInterface;
use Magento\GiftCardAccount\Api\Data\GiftCardAccountInterface;
use Magento\GiftCardAccount\Api\GiftCardAccountRepositoryInterface;
use Magento\GiftCardAccount\Model\Giftcardaccount;
use Magento\GiftCardAccount\Model\GiftcardaccountFactory;
use Magento\GiftCardAccount\Model\History;
use Magento\Store\Model\StoreManagerInterface;
use Psr\Log\LoggerInterface;

/**
 * Accredo gift card processor implementation
 */
class Processor implements AccredoProcessorInterface
{
    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var GiftCardAccountRepositoryInterface
     */
    private $giftCardRepository;

    /**
     * @var SearchCriteriaBuilder
     */
    private $searchCriteriaBuilder;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var GiftcardaccountFactory
     */
    private $giftcardaccountFactory;

    /**
     * @var Config
     */
    private $config;

    /**
     * @var Curl
     */
    private $httpClient;

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @param LoggerInterface $logger
     * @param GiftCardAccountRepositoryInterface $giftCardRepository
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param StoreManagerInterface $storeManager
     * @param GiftcardaccountFactory $giftcardaccountFactory
     * @param Config $config
     * @param Curl $httpClient
     * @param SerializerInterface $serializer
     */
    public function __construct(
        LoggerInterface $logger,
        GiftCardAccountRepositoryInterface $giftCardRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        StoreManagerInterface $storeManager,
        GiftcardaccountFactory $giftcardaccountFactory,
        Config $config,
        Curl $httpClient,
        SerializerInterface $serializer
    ) {
        $this->logger = $logger;
        $this->giftCardRepository = $giftCardRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->storeManager = $storeManager;
        $this->giftcardaccountFactory = $giftcardaccountFactory;
        $this->config = $config;
        $this->httpClient = $httpClient;
        $this->serializer = $serializer;
    }

    /**
     * @inheritDoc
     */
    public function syncGiftcard(string $code, ?int $websiteId = null): bool
    {


        try {
            // Step 1: Fetch gift card data from Accredo
            $accredoData = $this->fetchFromAccredo($code);
            
            // If card doesn't exist in Accredo, do nothing
            if (!$accredoData) {
                $this->logger->info('BrittainWynyard_GiftcardAccount: Gift card not found in Accredo, skipping sync', [
                    'code' => $code
                ]);
                return true;
            }

            // Step 2: Try to find the gift card in Magento
            $magentoGiftCard = $this->findGiftCardInMagento($code);
            
            // Step 3: Apply business logic based on scenarios
            if ($magentoGiftCard) {
                // Card exists in Magento
                return $this->handleExistingGiftCard($magentoGiftCard, $accredoData, $code);
            } else {
                // Card doesn't exist in Magento but exists in Accredo - create new
                return $this->createNewGiftCard($accredoData, $websiteId);
            }
            
        } catch (\Exception $e) {
            $this->logger->error('BrittainWynyard_GiftcardAccount: Exception during gift card sync', [
                'code' => $code,
                'websiteId' => $websiteId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            throw new LocalizedException(
                __('Failed to sync gift card with Accredo: %1', $e->getMessage()),
                $e
            );
        }
    }

    /**
     * Fetch gift card data from Accredo API
     *
     * Makes real HTTP request if configured, falls back to mock data otherwise
     *
     * @param string $code
     * @return array|null
     */
    private function fetchFromAccredo(string $code): ?array
    {
        $websiteId = $this->storeManager->getStore()->getWebsiteId();
        
        // Check if Accredo integration is enabled and configured
        if (!$this->config->isAccredoEnabled($websiteId)) {
            return null;
        }

        // Always attempt real API call (mock behavior controlled in override method)
        try {
            return $this->callAccredoApi($code, $websiteId);
        } catch (\Exception $e) {
            $this->logger->error('BrittainWynyard_GiftcardAccount: Accredo API call failed', [
                'code' => $code,
                'error' => $e->getMessage()
            ]);
            
            // No fallback - let the failure propagate for proper error handling
            return null;
        }
    }

    /**
     * Make actual HTTP request to Accredo API
     *
     * @param string $code
     * @param int|null $websiteId
     * @return array|null
     * @throws \Exception
     */
    private function callAccredoApi(string $code, ?int $websiteId = null): ?array
    {
        $endpoint = $this->config->getAccredoEndpoint($websiteId);
        $bearerToken = $this->config->getAccredoBearerToken($websiteId);
        $timeout = $this->config->getRequestTimeout($websiteId);

        if (empty($endpoint) || empty($bearerToken)) {
            throw new LocalizedException(__('Accredo API endpoint or bearer token not configured'));
        }

        // Prepare URL with query parameters
        $queryParams = http_build_query([
            'cardno' => $code
        ]);
        $endpointWithParams = $endpoint . '?' . $queryParams;
   
        // Configure HTTP client
        $this->httpClient->setTimeout($timeout);
        $this->httpClient->setHeaders([
            'Authorization' => 'Bearer ' . $bearerToken,
            'x-brand' => 'asics',
            'Accept' => '*/*'
        ]);


        // Make GET request
        $this->httpClient->get($endpointWithParams);

        $httpStatus = $this->httpClient->getStatus();
        $responseBody = $this->httpClient->getBody();



        // Handle different HTTP status codes
        switch ($httpStatus) {
            case 200:
                // Success - parse and return the data
                return $this->parseAccredoResponse($responseBody, $code);
                
            case 404:
                // Gift card not found in Accredo
                return null;
                
            default:
                // Other error responses
                throw new \Exception(sprintf(
                    'Accredo API returned unexpected status %d: %s',
                    $httpStatus,
                    $responseBody
                ));
        }
    }



    /**
     * Parse Accredo API response into normalized format
     *
     * @param string $responseBody
     * @param string $code
     * @return array|null
     */
    private function parseAccredoResponse(string $responseBody, string $code): ?array
    {
        try {
            $responseData = $this->serializer->unserialize($responseBody);
            


            // Handle new nested data format - extract data from 'data' property if present
            $actualData = $responseData;
            if (isset($responseData['data']) && is_array($responseData['data'])) {
                $actualData = $responseData['data'];
            }

            // Map Accredo response fields to our normalized format
            $parsedData = [
                'code' => $code,
                'balance' => (float)($actualData['card_balance'] ?? $actualData['balance'] ?? 0),
                'status' => 1, // Default to active
                'expires' => $this->parseAccredoDate($actualData['issuedate'] ?? null),
                'recipient_email' => '',
                'recipient_name' => '',
                'card_no' => $actualData['card_no'] ?? null,
                'cardid' => $actualData['cardid'] ?? null,
                'modified_date' => $actualData['ModifiedDate'] ?? null,
                'modified_user' => $actualData['ModifiedUserCode'] ?? null,
                'accredo_data' => $actualData // Store processed data for reference
            ];
            
            // Log the code/balance received from Accredo
            $this->logger->info('BrittainWynyard_GiftcardAccount: Received card data from Accredo', [
                'code' => $code,
                'accredo_balance' => $parsedData['balance']
            ]);
            
            return $parsedData;
            
        } catch (\Exception $e) {
            $this->logger->error('BrittainWynyard_GiftcardAccount: Failed to parse Accredo response', [
                'code' => $code,
                'response_body' => $responseBody,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Parse Accredo date format into Magento-compatible date
     *
     * @param string|null $accredoDate
     * @return string
     */
    private function parseAccredoDate(?string $accredoDate): string
    {
        if (!$accredoDate) {
            // Default to 1 year from now if no date provided
            return date('Y-m-d', strtotime('+1 year'));
        }

        try {
            // Accredo uses YYYY-MM-DD format, convert to ensure it's valid
            $timestamp = strtotime($accredoDate);
            if ($timestamp === false || $timestamp < time()) {
                // If invalid date or past date, default to 1 year from now
                return date('Y-m-d', strtotime('+1 year'));
            }
            return date('Y-m-d', $timestamp);
        } catch (\Exception $e) {
            // Fallback to 1 year from now
            return date('Y-m-d', strtotime('+1 year'));
        }
    }

    /**
     * Find gift card in Magento by code
     *
     * @param string $code
     * @return Giftcardaccount|null
     */
    private function findGiftCardInMagento(string $code): ?Giftcardaccount
    {
        try {
            /** @var Giftcardaccount $giftCard */
            $giftCard = $this->giftcardaccountFactory->create();
            $giftCard->loadByCode($code);

            if ($giftCard->getId()) {
                $this->logger->info('BrittainWynyard_GiftcardAccount: Gift card found in Magento', [
                    'code' => $code,
                    'id' => $giftCard->getId(),
                    'current_balance' => $giftCard->getBalance()
                ]);
                return $giftCard;
            }


            return null;

        } catch (\Exception $e) {
            $this->logger->error('BrittainWynyard_GiftcardAccount: Error searching for gift card in Magento', [
                'code' => $code,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Handle existing gift card - only decrease balance if Accredo balance is lower
     *
     * @param Giftcardaccount $magentoGiftCard
     * @param array $accredoData
     * @param string $code
     * @return bool
     */
    private function handleExistingGiftCard(Giftcardaccount $magentoGiftCard, array $accredoData, string $code): bool
    {
        $currentBalance = (float)$magentoGiftCard->getBalance();
        $accredoBalance = $accredoData['balance'];

        // Compare balances (with small tolerance for float comparison)
        if (abs($currentBalance - $accredoBalance) < 0.01) {
            $this->logger->info('BrittainWynyard_GiftcardAccount: Gift card balances match, no update needed', [
                'code' => $code,
                'balance' => $currentBalance
            ]);
            return true;
        }

        // Only decrease balance - if Accredo balance is higher, don't update
        // The main reason we are doing this is to prevent the balance from being increased if the Accredo balance if the order has been placed recently and accredo has not been updated yet.
        // We have a background job that updates the accredo balance that will handle cases where the balance has genuinely increased.
        if ($accredoBalance > $currentBalance) {
            $this->logger->info('BrittainWynyard_GiftcardAccount: Accredo balance is higher than Magento balance, skipping update', [
                'code' => $code,
                'magento_balance' => $currentBalance,
                'accredo_balance' => $accredoBalance
            ]);
            return true;
        }

        // Accredo balance is lower - update the gift card to decrease balance
        $this->logger->info('BrittainWynyard_GiftcardAccount: Gift card balance will be decreased', [
            'code' => $code,
            'magento_balance' => $currentBalance,
            'accredo_balance' => $accredoBalance
        ]);

        try {
            // Set custom additional info for Accredo sync
            $additionalInfo = sprintf(
                'Accredo Sync: Previous $%s, Accredo $%s, Updated to $%s (Delta: -$%s)',
                number_format($currentBalance, 2),
                number_format($accredoBalance, 2),
                number_format($accredoBalance, 2),
                number_format($currentBalance - $accredoBalance, 2)
            );

            // Set history action and additional info on the gift card itself
            $magentoGiftCard->setHistoryAction(History::ACTION_UPDATED);
            $magentoGiftCard->setBalanceDelta($accredoBalance - $currentBalance);
            $magentoGiftCard->setData('history_additional_info', $additionalInfo);
            
            // Update the gift card balance - this will trigger history creation
            $magentoGiftCard->setBalance($accredoBalance);
            $magentoGiftCard->save();
            
            $this->logger->info('BrittainWynyard_GiftcardAccount: Gift card balance decreased successfully', [
                'code' => $code,
                'new_balance' => $accredoBalance,
                'delta' => $accredoBalance - $currentBalance
            ]);
            return true;
            
        } catch (\Exception $e) {
            $this->logger->error('BrittainWynyard_GiftcardAccount: Failed to update gift card balance', [
                'code' => $code,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Create new gift card in Magento from Accredo data
     *
     * @param array $accredoData
     * @param int|null $websiteId
     * @return bool
     */
    private function createNewGiftCard(array $accredoData, ?int $websiteId = null): bool
    {


        try {
            // Use website ID from parameter or default to current website
            if ($websiteId === null) {
                $websiteId = $this->storeManager->getWebsite()->getId();
            }

            /** @var Giftcardaccount $newGiftCard */
            $newGiftCard = $this->giftcardaccountFactory->create();
            
            // Set required gift card data from Accredo
            $newGiftCard->setCode($accredoData['code']);
            $newGiftCard->setBalance((float)$accredoData['balance']);
            $newGiftCard->setStatus($accredoData['status'] ?? Giftcardaccount::STATUS_ENABLED);
            $newGiftCard->setWebsiteId($websiteId);
            $newGiftCard->setState(Giftcardaccount::STATE_AVAILABLE);
            $newGiftCard->setIsRedeemable(Giftcardaccount::REDEEMABLE);

            // Set expiration date in proper Y-m-d format if provided
            if (!empty($accredoData['expires'])) {
                // Ensure date is in Y-m-d format and is in the future
                $expiryDate = $accredoData['expires'];
                if (strtotime($expiryDate) > time()) {
                    $newGiftCard->setDateExpires(date('Y-m-d', strtotime($expiryDate)));
                } else {
                    // If date is in the past, set to 1 year from now
                    $newGiftCard->setDateExpires(date('Y-m-d', strtotime('+1 year')));
                    $this->logger->warning(
                        'BrittainWynyard_GiftcardAccount: Accredo expiry date was in past, setting to +1 year',
                        [
                        'code' => $accredoData['code'],
                        'original_date' => $expiryDate,
                        'new_date' => date('Y-m-d', strtotime('+1 year'))
                        ]
                    );
                }
            } else {
                // Default to 1 year expiry if no date provided
                $newGiftCard->setDateExpires(date('Y-m-d', strtotime('+1 year')));
            }

            // Set additional optional data if available
            if (!empty($accredoData['recipient_email'])) {
                $newGiftCard->setRecipientEmail($accredoData['recipient_email']);
            }
            if (!empty($accredoData['recipient_name'])) {
                $newGiftCard->setRecipientName($accredoData['recipient_name']);
            }



            // Set custom additional info for creation history
            $creationInfo = sprintf(
                'Created from Accredo sync with initial balance of $%s',
                number_format($accredoData['balance'], 2)
            );
            $newGiftCard->setData('history_additional_info', $creationInfo);

            // Save the new gift card - this will automatically create history entry
            $newGiftCard->save();
            
            $this->logger->info('BrittainWynyard_GiftcardAccount: New gift card created successfully', [
                'code' => $accredoData['code'],
                'id' => $newGiftCard->getId(),
                'balance' => $accredoData['balance']
            ]);

            return true;

        } catch (\Magento\Framework\Exception\CouldNotSaveException $e) {
            $this->logger->error('BrittainWynyard_GiftcardAccount: Could not save gift card account', [
                'code' => $accredoData['code'],
                'error' => $e->getMessage(),
                'previous_error' => $e->getPrevious() ? $e->getPrevious()->getMessage() : null
            ]);

            // Try to get more specific error information
            if (strpos($e->getMessage(), 'duplicate') !== false || strpos($e->getMessage(), 'Duplicate') !== false) {
                $this->logger->warning(
                    'BrittainWynyard_GiftcardAccount: Gift card code may already exist, ' .
                    'attempting to update existing card',
                    [
                    'code' => $accredoData['code']
                    ]
                );
                
                // Try to find and update existing card instead
                $existingCard = $this->findGiftCardInMagento($accredoData['code']);
                if ($existingCard) {
                    return $this->handleExistingGiftCard($existingCard, $accredoData, $accredoData['code']);
                }
            }
            
            return false;
        } catch (\Exception $e) {
            $this->logger->error('BrittainWynyard_GiftcardAccount: Failed to create new gift card', [
                'code' => $accredoData['code'],
                'error' => $e->getMessage(),
                'error_class' => get_class($e),
                'trace' => $e->getTraceAsString()

            ]);
            return false;
        }
    }
}