<?php

declare(strict_types=1);

namespace BrittainWynyard\Searchspring\Model\Feed\DataProvider;

use Magento\Store\Model\StoreManagerInterface;
use SearchSpring\Feed\Api\Data\FeedSpecificationInterface;
use SearchSpring\Feed\Model\Feed\DataProvider\Stock\StockResolverInterface;
use SearchSpring\Feed\Model\Feed\DataProviderInterface;

/**
 * Adds the child product stock level data
 */
class ChildStockProvider implements DataProviderInterface
{
    public function __construct(
        private readonly StockResolverInterface $stockResolver,
        private readonly StoreManagerInterface  $storeManager,
    ) {
    }

    /**
     * @inheritDoc
     */
    public function getData(array $products, FeedSpecificationInterface $feedSpecification): array
    {
        if (!$feedSpecification->getIncludeJSONConfig()) {
            return $products;
        }

        $productIds = [];
        foreach ($products as $product) {
            $jsonConfig = $this->extractDecodedJsonConfig($product);
            if (!$jsonConfig) {
                continue;
            }

            $childIds = $this->extractChildIds($jsonConfig);
            if (!$childIds) {
                continue;
            }

            array_push($productIds, ...$childIds);
        }

        $storeId = (int)$this->storeManager->getStore($feedSpecification->getStoreCode())->getId();
        $stockData = $this->stockResolver->resolve()->getStock($productIds, $storeId);

        foreach ($products as &$product) {
            $jsonConfig = $this->extractDecodedJsonConfig($product);
            if (!$jsonConfig) {
                continue;
            }

            $childIds = $this->extractChildIds($jsonConfig);
            if (!$childIds) {
                continue;
            }

            $childStock = [];
            foreach ($childIds as $childId) {
                if (!isset($stockData[$childId])) {
                    continue;
                }

                $childStock[$childId] = $stockData[$childId];
            }

            $jsonConfig['child_stock'] = $childStock;

            $product['json_config'] = json_encode($jsonConfig);
        }

        return $products;
    }

    /**
     * Extract decoded json_config from product data array
     *
     * @param array $product
     * @return array|null
     */
    private function extractDecodedJsonConfig(array $product): ?array
    {
        if (!isset($product['json_config']) || !$product['json_config']) {
            return null;
        }

        return json_decode($product['json_config'], true);
    }

    /**
     * Extract child IDs from product json_config data array
     *
     * @param array $jsonConfig
     * @return int[]|null
     */
    private function extractChildIds(array $jsonConfig): ?array
    {
        if (!isset($jsonConfig['index'])) {
            return null;
        }

        return array_keys($jsonConfig['index']);
    }

    /**
     * @inheritDoc
     */
    public function reset(): void
    {
        // do nothing
    }

    /**
     * @inheritDoc
     */
    public function resetAfterFetchItems(): void
    {
        // do nothing
    }
}
