<?php

namespace Overdose\InStorePickup\Controller\StockInfo;

use Magento\Framework\App\Action\Action;
use Magento\Framework\App\Action\Context;
use Magento\Framework\Controller\Result\JsonFactory;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\InputException;
use Overdose\InStorePickup\ViewModel\StockInfo;
use Magento\Framework\View\LayoutFactory;
use Magento\Framework\View\Result\PageFactory;
use Psr\Log\LoggerInterface;

class Ajax extends Action
{
    /**
     * @var JsonFactory
     */
    protected $jsonFactory;

    /**
     * @var ProductRepositoryInterface
     */
    protected $productRepository;

    /**
     * @var StockInfo
     */
    protected $stockInfoViewModel;

    /**
     * @var LayoutFactory
     */
    protected $layoutFactory;

    /**
     * @var PageFactory
     */
    protected $pageFactory;

    /**
     * @var LoggerInterface
     */
    protected $logger;

    /**
     * @param Context $context
     * @param JsonFactory $jsonFactory
     * @param ProductRepositoryInterface $productRepository
     * @param StockInfo $stockInfoViewModel
     * @param LayoutFactory $layoutFactory
     * @param PageFactory $pageFactory
     * @param LoggerInterface $logger
     */
    public function __construct(
        Context $context,
        JsonFactory $jsonFactory,
        ProductRepositoryInterface $productRepository,
        StockInfo $stockInfoViewModel,
        LayoutFactory $layoutFactory,
        PageFactory $pageFactory,
        LoggerInterface $logger
    ) {
        parent::__construct($context);
        $this->jsonFactory = $jsonFactory;
        $this->productRepository = $productRepository;
        $this->stockInfoViewModel = $stockInfoViewModel;
        $this->layoutFactory = $layoutFactory;
        $this->pageFactory = $pageFactory;
        $this->logger = $logger;
    }

    /**
     * Execute AJAX request for stock info
     *
     * @return \Magento\Framework\Controller\Result\Json
     */
    public function execute()
    {
        $result = $this->jsonFactory->create();
        
        try {
            $productId = $this->getRequest()->getParam('product_id');
            
            if (!$productId) {
                throw new \Magento\Framework\Exception\InputException(__('Product ID is required'));
            }

            // Load the product
            $product = $this->productRepository->getById($productId);
            
            // Set the product in the view model
            $this->stockInfoViewModel->setProduct($product);
            
            // Get the stock data - decode JSON strings to return as objects
            $stockData = [
                'colourAttr' => json_decode($this->stockInfoViewModel->getInitColorAttributes(), true),
                'msiList' => json_decode($this->stockInfoViewModel->getMsiList(), true),
                'msiStock' => json_decode($this->stockInfoViewModel->getInitMsiStock(), true),
                'initMsiStoreSize' => json_decode($this->stockInfoViewModel->getInitMsiStoreSize(), true),
                'popupModelContent' => json_decode($this->stockInfoViewModel->getSizeAvailableContent(), true),
                'productId' => $productId
            ];

            $result->setData([
                'success' => true,
                'data' => $stockData
            ]);

        } catch (NoSuchEntityException $e) {
            $result->setData([
                'success' => false,
                'message' => 'Product not found'
            ]);
        } catch (InputException $e) {
            $result->setData([
                'success' => false,
                'message' => 'Invalid input parameters'
            ]);
        } catch (\Exception $e) {
            // Log the actual error for debugging
            $this->logger->error('InStorePickup AJAX Error: ' . $e->getMessage(), [
                'exception' => $e,
                'product_id' => $this->getRequest()->getParam('product_id')
            ]);
            
            $result->setData([
                'success' => false,
                'message' => 'Failed to load inventory data'
            ]);
        }

        return $result;
    }
}
