<?php

namespace Overdose\InStorePickup\Model\Checkout;

use Magento\Checkout\Model\Session;
use Magento\InventorySalesApi\Api\Data\SalesChannelInterface;
use Magento\InventorySalesApi\Api\StockResolverInterface;
use Magento\InventorySalesApi\Model\GetStockItemDataInterface;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Inventory\Model\ResourceModel\Source\CollectionFactory as SourceCollectionFactory;
use Magento\Inventory\Model\Source\Command\GetSourcesAssignedToStockOrderedByPriority;
use Magento\Inventory\Model\ResourceModel\SourceItem\CollectionFactory as SourceItemCollectionFactory;

class ConfigProvider
{
    /**
     * @var CartRepositoryInterface
     */
    private $cartRepository;

    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var StockResolverInterface
     */
    protected $stockResolver;

    /**
     * @var GetStockItemDataInterface
     */
    private $getStockItemData;

    /**
     * @var SourceCollectionFactory
     */
    protected $sourceCollectionFactory;

    /**
     * @var GetSourcesAssignedToStockOrderedByPriority
     */
    protected $getSourcesAssignedToStockOrderedByPriority;

    /**
     * @var SourceItemCollectionFactory
     */
    protected $sourceItemCollectionFactory;

    /**
     * @var Session
     */
    protected $checkoutSession;

    /**
     * @param CartRepositoryInterface $cartRepository
     * @param StoreManagerInterface $storeManager
     * @param StockResolverInterface $stockResolver
     * @param GetStockItemDataInterface $getStockItemData
     * @param SourceCollectionFactory $sourceCollectionFactory
     * @param SourceItemCollectionFactory $sourceItemCollectionFactory
     * @param Session $checkoutSession
     * @param GetSourcesAssignedToStockOrderedByPriority $getSourcesAssignedToStockOrderedByPriority
     */
    public function __construct(
        CartRepositoryInterface                    $cartRepository,
        StoreManagerInterface                      $storeManager,
        StockResolverInterface                     $stockResolver,
        GetStockItemDataInterface                  $getStockItemData,
        SourceCollectionFactory                    $sourceCollectionFactory,
        SourceItemCollectionFactory                $sourceItemCollectionFactory,
        Session                                    $checkoutSession,
        GetSourcesAssignedToStockOrderedByPriority $getSourcesAssignedToStockOrderedByPriority
    ) {
        $this->cartRepository = $cartRepository;
        $this->storeManager = $storeManager;
        $this->stockResolver = $stockResolver;
        $this->getStockItemData = $getStockItemData;
        $this->sourceCollectionFactory = $sourceCollectionFactory;
        $this->sourceItemCollectionFactory = $sourceItemCollectionFactory;
        $this->checkoutSession = $checkoutSession;
        $this->getSourcesAssignedToStockOrderedByPriority = $getSourcesAssignedToStockOrderedByPriority;
    }

    /**
     * @return array[]
     */
    public function getConfig()
    {
        $inventoryData = $this->getStoresInventoryData();
        
        return [
            'overdoseInStorePickup' => [
                'is_available_method' => $this->isAvailableMethod(),
                'stores_inventory_data' => $inventoryData['stores_inventory'],
                'available_stores' => $inventoryData['available_stores']
            ]
        ];
    }

    /**
     * @return bool
     */
    private function isAvailableMethod()
    {
        $quoteId = (int)$this->checkoutSession->getQuoteId();
        try {
            $listItemSku = [];
            $listInStorePickupSource = [];
            $listProductStockData = [];
            $website = $this->storeManager->getWebsite();
            $stock = $this->stockResolver->execute(SalesChannelInterface::TYPE_WEBSITE, $website->getCode());
            $stockId = (int)$stock->getStockId();
            $sourceList = $this->getSourcesAssignedToStockOrderedByPriority->execute($stockId);
            foreach ($sourceList as $source) {
                $extensionAttributes = $source->getExtensionAttributes();
                if ($extensionAttributes && 
                    method_exists($extensionAttributes, 'getIsPickupLocationActive') &&
                    $extensionAttributes->getIsPickupLocationActive() && 
                    $source->isEnabled()) {
                    $listInStorePickupSource[] = $source->getSourceCode();
                }
            }
            $cart = $this->cartRepository->get($quoteId);
            $cartItems = $cart->getItems();
            $cartItemRequirements = []; // Track required quantities per SKU
            foreach ($cartItems as $item) {
                $sku = $item->getSku();
                $requiredQty = (float)$item->getQty();
                $listItemSku[] = $sku;
                
                // Accumulate quantities for same SKU
                if (isset($cartItemRequirements[$sku])) {
                    $cartItemRequirements[$sku] += $requiredQty;
                } else {
                    $cartItemRequirements[$sku] = $requiredQty;
                }
            }
            
            $sourceItemCollection = $this->sourceItemCollectionFactory
                ->create()
                ->addFieldToFilter('sku', ['in' => $listItemSku])
                ->addFieldToFilter('source_code', ['in' => $listInStorePickupSource]);
                
            foreach ($sourceItemCollection as $sourceItem) {
                $sourceCode = $sourceItem->getSourceCode();
                $sku = $sourceItem->getSku();
                $availableQty = (float)$sourceItem->getQuantity();
                $isEnabled = (int)$sourceItem->getStatus() == 1;
                $requiredQty = $cartItemRequirements[$sku] ?? 0;
                
                // Check if this source has sufficient quantity for this SKU
                $hasSufficientQty = $isEnabled && $availableQty >= $requiredQty;
                
                if (!isset($listProductStockData[$sourceCode])) {
                    $listProductStockData[$sourceCode] = [];
                }
                $listProductStockData[$sourceCode][$sku] = $hasSufficientQty;
            }

            // Check if at least one store can fulfill all items with required quantities
            foreach ($listInStorePickupSource as $inStorePickupSource) {
                $canFulfillAllItems = true;
                foreach ($listItemSku as $sku) {
                    if (!isset($listProductStockData[$inStorePickupSource][$sku]) || 
                        !$listProductStockData[$inStorePickupSource][$sku]) {
                        $canFulfillAllItems = false;
                        break;
                    }
                }
                if ($canFulfillAllItems) {
                    return true;
                }
            }
            return false;
        } catch (\Exception $e) {
            return false;
        }
        return true;
    }

    /**
     * Get inventory data for all stores
     * @return array
     */
    private function getStoresInventoryData()
    {
        $quoteId = (int)$this->checkoutSession->getQuoteId();
        $storesInventory = [];
        $availableStores = [];

        try {
            $listItemSku = [];
            $listInStorePickupSource = [];
            $listProductStockData = [];
            $website = $this->storeManager->getWebsite();
            $stock = $this->stockResolver->execute(SalesChannelInterface::TYPE_WEBSITE, $website->getCode());
            $stockId = (int)$stock->getStockId();
            $sourceList = $this->getSourcesAssignedToStockOrderedByPriority->execute($stockId);

            // Get pickup location sources
            foreach ($sourceList as $source) {
                $extensionAttributes = $source->getExtensionAttributes();
                if ($extensionAttributes && 
                    method_exists($extensionAttributes, 'getIsPickupLocationActive') &&
                    $extensionAttributes->getIsPickupLocationActive() && 
                    $source->isEnabled()) {
                    $listInStorePickupSource[] = $source->getSourceCode();
                }
            }

            $cart = $this->cartRepository->get($quoteId);
            $cartItems = $cart->getItems();
            $cartItemRequirements = []; // Track required quantities per SKU
            foreach ($cartItems as $item) {
                $sku = $item->getSku();
                $requiredQty = (float)$item->getQty();
                $listItemSku[] = $sku;
                
                // Accumulate quantities for same SKU (in case of multiple cart items with same SKU)
                if (isset($cartItemRequirements[$sku])) {
                    $cartItemRequirements[$sku] += $requiredQty;
                } else {
                    $cartItemRequirements[$sku] = $requiredQty;
                }
            }

            // Get stock data for all combinations
            $sourceItemCollection = $this->sourceItemCollectionFactory
                ->create()
                ->addFieldToFilter('sku', ['in' => $listItemSku])
                ->addFieldToFilter('source_code', ['in' => $listInStorePickupSource]);
            
            foreach ($sourceItemCollection as $sourceItem) {
                $sourceCode = $sourceItem->getSourceCode();
                $sku = $sourceItem->getSku();
                $availableQty = (float)$sourceItem->getQuantity();
                $isEnabled = (int)$sourceItem->getStatus() == 1;
                
                if (!isset($listProductStockData[$sourceCode])) {
                    $listProductStockData[$sourceCode] = [];
                }
                
                // Store detailed quantity information instead of just boolean
                $listProductStockData[$sourceCode][$sku] = [
                    'available_qty' => $availableQty,
                    'is_enabled' => $isEnabled,
                    'sufficient_qty' => $isEnabled && $availableQty >= ($cartItemRequirements[$sku] ?? 0)
                ];
            }

            // For each store, check if ALL items have sufficient quantities
            foreach ($listInStorePickupSource as $sourceCode) {
                $allItemsAvailable = true;
                $storeInventoryStatus = [];
                
                foreach ($listItemSku as $sku) {
                    $requiredQty = $cartItemRequirements[$sku] ?? 0;
                    $stockData = $listProductStockData[$sourceCode][$sku] ?? null;
                    
                    if ($stockData) {
                        $availableQty = $stockData['available_qty'];
                        $isEnabled = $stockData['is_enabled'];
                        $hasSufficientQty = $stockData['sufficient_qty'];
                        
                        $storeInventoryStatus[$sku] = [
                            'required_qty' => $requiredQty,
                            'available_qty' => $availableQty,
                            'is_enabled' => $isEnabled,
                            'sufficient_qty' => $hasSufficientQty
                        ];
                        
                        if (!$hasSufficientQty) {
                            $allItemsAvailable = false;
                        }
                    } else {
                        // Item not found in this store
                        $storeInventoryStatus[$sku] = [
                            'required_qty' => $requiredQty,
                            'available_qty' => 0,
                            'is_enabled' => false,
                            'sufficient_qty' => false
                        ];
                        $allItemsAvailable = false;
                    }
                }
                
                $storesInventory[$sourceCode] = [
                    'all_items_available' => $allItemsAvailable,
                    'items_status' => $storeInventoryStatus
                ];
                
                if ($allItemsAvailable) {
                    $availableStores[] = $sourceCode;
                }
            }

        } catch (\Exception $e) {
            // In case of error, return empty data
            $storesInventory = [];
            $availableStores = [];
        }

        return [
            'stores_inventory' => $storesInventory,
            'available_stores' => $availableStores
        ];
    }
}
