<?php

namespace Verifone\Hosted\Helper;

use VerifoneEcomAPI\ApiWrapper\Http\ApiClient\Client;
use VerifoneEcomAPI\ApiWrapper\Http\SimpleCurl;
use VerifoneEcomAPI\ApiWrapper\Schemas\CaptureSchema;
use VerifoneEcomAPI\ApiWrapper\Schemas\CheckoutSchema;
use VerifoneEcomAPI\ApiWrapper\Schemas\CustomerSchema;
use VerifoneEcomAPI\ApiWrapper\Schemas\RefundSchema;
use VerifoneEcomAPI\ApiWrapper\Authentication\BasicAuth;
use VerifoneEcomAPI\ApiWrapper\Regions\Emea;
use VerifoneEcomAPI\ApiWrapper\Regions\Na;
use VerifoneEcomAPI\ApiWrapper\Regions\NzAu;
use VerifoneEcomAPI\ApiWrapper\Settings;

use Magento\Sales\Model\Order;
use Magento\Framework\App\CacheInterface;
use Exception;
use VerifoneEcommApi\ApiWrapper\Http\HttpException;

class Verifone extends \Magento\Framework\App\Helper\AbstractHelper
{
    const LOCALE_NL_CODES = ['nl_BE', 'nl_NL'];
    const LOCALE_NO_CODES = ['nb_NO', 'nn_NO'];

    const ZERO_DECIMAL_CURRENCIES = ['bif', 'clp', 'djf', 'gnf', 'isk', 'jpy', 'kmf', 'krw', 'pyg', 'rwf', 'ugx', 'uyi',
        'vnd', 'vuv', 'xaf', 'xof', 'xpf'];
    const VERIFONE_ENV_CAPTURE_EVENT_TYPE_STATUS_AUTHORIZED = 'AUTHORIZED';
    const VERIFONE_ENV_CAPTURE_EVENT_TYPE_STATUS_SETTLEMENT_COMPLETED = 'SETTLEMENT_COMPLETED';
    const VERIFONE_ENV_CAPTURE_EVENT_TYPE_STATUS_VOIDED = 'VOIDED';

    #webhook
    const VERIFONE_TRANSACTION_AUTHORIZED = 'AUTHORIZED';
    const VERIFONE_TRANSACTION_AUTHORISED = 'AUTHORISED';
    const VERIFONE_TRANSACTION_CAPTURE_SETTLED = 'CAPTURE SETTLED';
    const VERIFONE_TRANSACTION_SALE_AUTHORISED = 'SALE AUTHORISED';
    const VERIFONE_TRANSACTION_SALE_SETTLED = 'SALE SETTLED';
    const VERIFONE_TRANSACTION_AUTHORISATION_AUTHORISED = 'AUTHORISATION AUTHORISED';
    const VERIFONE_TRANSACTION_CAPTURE_AUTHORISED = 'CAPTURE AUTHORISED';
    const VERIFONE_TRANSACTION_PENDING = 'PENDING';
    const VERIFONE_TRANSACTION_REFUND_AUTHORISED = 'REFUND AUTHORISED';
    const VERIFONE_TRANSACTION_REFUND_REFUNDED = 'REFUND REFUNDED';
    const VERIFONE_TRANSACTION_REFUND_SETTLED = 'REFUND SETTLED';
    const VERIFONE_TRANSACTION_REFUNDED = 'REFUNDED';
    const VERIFONE_TRANSACTION_INITIATED = 'INITIATED';
    const VERIFONE_TRANSACTION_REFUND_SETTLEMENT_COMPLETED = 'SETTLEMENT_COMPLETED';
    const VERIFONE_TRANSACTION_CARD_VERIFICATION_AUTHORISED = 'CARD_VERIFICATION AUTHORISED';

    const VERIFONE_WEBHOOK_EVENT_TYPE_AUTHORISATION_APPROVED = 'TxnAuthorisationApproved';
    const VERIFONE_WEBHOOK_EVENT_TYPE_AUTHORISATION_DECLINED = 'TxnAuthorisationDeclined';
    const VERIFONE_WEBHOOK_EVENT_TYPE_CAPTURE_APPROVED = 'TxnCaptureApproved';
    const VERIFONE_WEBHOOK_EVENT_TYPE_CAPTURE_DECLINED = 'TxnCaptureDeclined';
    const VERIFONE_WEBHOOK_EVENT_TYPE_REFUND_APPROVED = 'TxnRefundApproved';
    const VERIFONE_WEBHOOK_EVENT_TYPE_REFUND_DECLINED = 'TxnRefundDeclined';
    const VERIFONE_WEBHOOK_EVENT_TYPE_SALE_APPROVED = 'TxnSaleApproved';
    const VERIFONE_WEBHOOK_EVENT_TYPE_SALE_DECLINED = 'TxnSaleDeclined';
    #/webhook

    private $refundStatusOk = [
        self::VERIFONE_TRANSACTION_AUTHORIZED,
        self::VERIFONE_TRANSACTION_AUTHORISED,
        self::VERIFONE_TRANSACTION_INITIATED,
        self::VERIFONE_TRANSACTION_REFUNDED,
        self::VERIFONE_TRANSACTION_REFUND_SETTLEMENT_COMPLETED
    ];

    protected $_session;
    protected $_logger;

    /**
     * @var \Magento\Sales\Model\OrderFactory
     */
    protected $_orderFactory;

    /**
     * @var \Magento\Framework\Api\SearchCriteriaBuilder
     */
    protected $_searchCriteriaBuilder;

    /**
     * @var \Magento\Quote\Model\QuoteRepository
     */
    protected $_quoteRepo;

    /**
     * @var CacheInterface
     */
    private $_cache;

    /**
     * @var \Magento\Sales\Model\Service\InvoiceService
     */
    protected $_invoiceService;

    /**
     * @var \Magento\Framework\DB\TransactionFactory
     */
    protected $_transactionFactory;

    protected $_transactionBuilder;

    private $_verifoneSettings;
    private $_simpleCurl;
    private $_mapper;
    private $_client;


    public function __construct(
        \Magento\Framework\App\Helper\Context $context,
        \Magento\Checkout\Model\Session $session,
        \Magento\Framework\App\CacheInterface $cache,
        \Magento\Sales\Model\OrderFactory $orderFactory,
        \Magento\Framework\Api\SearchCriteriaBuilder $searchCriteriaBuilder,
        \Magento\Quote\Model\QuoteRepository $quoteRepo,
        \Magento\Sales\Model\Service\InvoiceService $invoiceService,
        \Magento\Framework\DB\TransactionFactory $transactionFactory,
        \Magento\Sales\Model\Order\Payment\Transaction\BuilderInterface $transactionBuilder
    ) {
        $this->_session               = $session;
        $this->_cache                 = $cache;
        $this->_logger                = $context->getLogger();
        $this->_orderFactory          = $orderFactory;
        $this->_searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->_quoteRepo             = $quoteRepo;
        $this->_invoiceService        = $invoiceService;
        $this->_transactionFactory    = $transactionFactory;
        $this->_transactionBuilder    = $transactionBuilder;
        parent::__construct($context);

        $this->_simpleCurl = new SimpleCurl();
    }

    /**
     * @param $comment
     *
     * @return bool
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function cancelCurrentOrder($comment)
    {
        $order = $this->_session->getLastRealOrder();
        if ($order->getId() && $order->getState() != \Magento\Sales\Model\Order::STATE_CANCELED) {
            $order->registerCancellation($comment)->save();

            return true;
        }

        return false;
    }

    /**
     * @return bool
     */
    public function restoreQuote()
    {
        return $this->_session->restoreQuote();
    }

    /**
     * @param       $route
     * @param  array  $params
     *
     * @return string
     */
    public function getUrl($route, $params = [])
    {
        return $this->_getUrl($route, $params);
    }

    /**
     * @param $incrementId
     * @param $refNo
     *
     * @return false|\Magento\Sales\Model\Order
     */
    public function getOrderByIncrementId($incrementId, $refNo = null)
    {
        $foundOrder = $this->_orderFactory->create()->loadByIncrementId($incrementId);
        if ($foundOrder->getId() && ( ! $refNo || $refNo == $foundOrder->getExtOrderId())) {
            return $foundOrder;
        }

        return false;
    }

    /**
     * @param $incrementId
     *
     * @return \Magento\Quote\Api\Data\CartInterface|null
     */
    public function getQuoteByIncrementId($incrementId)
    {
        $sc       = $this->_searchCriteriaBuilder->addFilter(
            \Magento\Quote\Api\Data\CartInterface::KEY_RESERVED_ORDER_ID,
            $incrementId
        )->create();
        $quoteArr = $this->_quoteRepo->getList($sc)->getItems();
        $quote    = null;

        if (count($quoteArr) > 0) {
            $quote = current($quoteArr);
        }

        return $quote;
    }

    /**
     * @param $paymentMethod
     */
    public function setUpVerifoneSettings($paymentMethod)
    {
//        echo json_encode($paymentMethod->getConfigData('alternative_settings'));die;
        $this->_verifoneSettings = new Settings();
        $this->_verifoneSettings
            ->setTest('1' == $paymentMethod->getConfigData('account_settings')['demo_mode'])
            ->setRegion($paymentMethod->getConfigData('account_settings')['region'])
            ->setApiUserId($paymentMethod->getConfigData('account_settings')['api_user'])
            ->setApiKey($paymentMethod->getConfigData('account_settings')['api_key'])
            ->setIsCardEnabled('1' == $paymentMethod->getConfigData('card_settings')['card_enabled'])
            ->set3dsEnable('1' == $paymentMethod->getConfigData('card_settings')['tds'])
            ->set3dsContractId($paymentMethod->getConfigData('card_settings')['tds_contract_id'] ?? '')
            ->setPaymentContractId($paymentMethod->getConfigData('card_settings')['payment_contract_id'])
            ->setEntityId($paymentMethod->getConfigData('account_settings')['entity_id'])
            ->setFraudCheck($paymentMethod->getConfigData('card_settings')['fraud_check'])
            ->setFraudCheckContractId($paymentMethod->getConfigData('card_settings')['fraud_check_contract_id'] ?? '')
            ->setTransactionType($paymentMethod->getConfigData('card_settings')['transaction_type'])
            ->setCurlVerifySsl('1' == $paymentMethod->getConfigData('account_settings')['verifone_curl'])
            ->setOrderStatus($paymentMethod->getConfigData('account_settings')['order_status'])
            ->setOrderBeforeRedirect($paymentMethod->getConfigData('account_settings')['order_before_redirect'])
            ->setPaymentType(Settings::PAYMENT_TYPE_HPP == $paymentMethod->getConfigData('display_settings')['payment_type']
                ? Settings::PAYMENT_TYPE_HPP
                : Settings::PAYMENT_TYPE_IFRAME)
            ->setVippsEnabled('1' == strtolower($paymentMethod->getConfigData('alternative_settings')['vipps']['vipps_enabled'] ?? ''))
            ->setVippsScaLevel($paymentMethod->getConfigData('alternative_settings')['vipps']['vipps_sca_compliance_level'] ?? 0)
            ->setVippsContractId($paymentMethod->getConfigData('alternative_settings')['vipps']['vipps_contract_id'] ?? '')
            ->setVipps3DSContractId($paymentMethod->getConfigData('alternative_settings')['vipps']['vipps_threeds_contract_id'] ?? '')

            ->setMobilePayEnabled('1' == strtolower($paymentMethod->getConfigData('alternative_settings')['mobile_pay']['mobile_pay_enabled'] ?? ''))
            ->setMobilePayScaLevel($paymentMethod->getConfigData('alternative_settings')['mobile_pay']['mobile_pay_sca_compliance_level'] ?? 0)
            ->setMobilePayContractId($paymentMethod->getConfigData('alternative_settings')['mobile_pay']['mobile_pay_contract_id'] ?? '')
            ->setMobilePay3DSContractId($paymentMethod->getConfigData('alternative_settings')['mobile_pay']['mobile_pay_threeds_contract_id'] ?? '')

            ->setApplePayScaComplianceLevel($paymentMethod->getConfigData('alternative_settings')['applepay']['apple_pay_sca_compliance_level'] ?? 0)
            ->setApplePayThreedsContractId($paymentMethod->getConfigData('alternative_settings')['applepay']['apple_pay_threeds_contract_id'] ?? '')
            ->setApplePayContractId($paymentMethod->getConfigData('alternative_settings')['applepay']['apple_pay_contract_id'] ?? '')
            ->setPaypalContractId($paymentMethod->getConfigData('alternative_settings')['paypal']['paypal_contract_id'] ?? '')
            ->setIsGooglePayEnabled('1' == strtolower($paymentMethod->getConfigData('alternative_settings')['googlepay']['google_pay_enabled'] ?? ''))
            ->setGooglePayContractId($paymentMethod->getConfigData('alternative_settings')['googlepay']['google_pay_contract_id']??'')
            ->setGooglePayScaComplianceLevel($paymentMethod->getConfigData('alternative_settings')['googlepay']['google_pay_sca_compliance_level'] ?? '')
            ->setGooglePayThreedsContractId($paymentMethod->getConfigData('alternative_settings')['googlepay']['google_pay_threeds_contract_id'] ?? '')
            ->setIsSwishEnabled('1' == strtolower($paymentMethod->getConfigData('alternative_settings')['swish']['swish_enabled'] ?? ''))
            ->setIsKlarnaEnabled('1' == strtolower($paymentMethod->getConfigData('alternative_settings')['klarna']['klarna_enabled'] ?? ''))
            ->setIsApplePayEnabled('1' == strtolower($paymentMethod->getConfigData('alternative_settings')['applepay']['apple_pay_enabled'] ?? ''))
            ->setIsPaypalEnabled('1' == strtolower($paymentMethod->getConfigData('alternative_settings')['paypal']['paypal_enabled'] ?? ''))
            ->setGpp2Enabled('1' == strtolower($paymentMethod->getConfigData('alternative_settings')['gpp2']['gpp2_enabled'] ?? ''))
            ->setGpp2ContractId($paymentMethod->getConfigData('alternative_settings')['gpp2']['gpp2_contract_id'] ?? '')
            ->setThemeId($paymentMethod->getConfigData('display_settings')['theme_id'] ?? '')
            ->setAbsInstallmentsEnabled($paymentMethod->getConfigData('card_settings')['abs_instalments_enabled'])
            ->setAbsInstallmentsThresholds($paymentMethod)
            ->setInvoice4uEnabled($paymentMethod->getConfigData('card_settings')['invoice_4u_enabled'] ?? false)
            ->setInvoice4uSendTaxRates($paymentMethod->getConfigData('card_settings')['invoice_4u_send_tax_rates'] ?? false)
            ->setInvoice4uReceiptType($paymentMethod->getConfigData('card_settings')['invoice_4u_receipt_type'] ?? Settings::INVOICE_4U_RECEIPT_TYPE_INVOICE)
        ;
    }

    public function getVerifoneSettings()
    {
        return $this->_verifoneSettings;
    }

    /**
     */
    public function getRegionMapper()
    {
        switch ($this->_verifoneSettings->getRegion()) {
            case 'NZAU':
                $this->_mapper = new Nzau($this->_verifoneSettings);
                break;
            case 'NA':
                $this->_mapper = new Na($this->_verifoneSettings);
                break;
            case 'EMEA':
                $this->_mapper = new Emea($this->_verifoneSettings);
                break;
            default:
                // will happen only on the first installation
                $this->_mapper = new Nzau($this->_verifoneSettings);
        }
    }

    /**
     * @param $order
     * @param $payment
     * @param $transactionId
     * @param $paymentMethod
     * @param  string  $type
     *
     * @return array|string[]|void
     * @throws \VerifoneEcomAPI\ApiWrapper\Http\HttpException
     */
    public function voidOrCapture($order, $payment, $transactionId, $paymentMethod, string $type = 'capture')
    {
        $this->setUpVerifoneSettings($paymentMethod);
        $this->getRegionMapper();
        $auth          = new BasicAuth($this->_verifoneSettings);
        $this->_client = new Client(
            $this->_mapper,
            $this->_verifoneSettings,
            $auth,
            $this->_simpleCurl
        );

        $amount = $this->convertAmountToVerifone($order->getGrandTotal(), $order->getOrderCurrencyCode());
        try {
            $payload = [
                'amount' => (int) str_replace([',', '.'], '', $amount),
            ];
            if ($type == 'void') {
                $result = $this->_client->postVoidAuthorization($payload, $transactionId);
            }
            if ($type == 'capture') {
                $result = $this->_client->postCapture($payload, $transactionId, new CaptureSchema());
            }

        } catch (HttpException $exception) {
            $this->_logger->critical(sprintf(
                'There was an error in the void capture, last known error was "%s", in file %s, at line %s. Error message: %s',
                json_last_error(),
                __FILE__,
                __LINE__,
                $exception->getMessage()
            ));

            return $this->failure();
        }

        if (isset($result['code']) && $result['code'] == 400) {
            return;
        }
        if (isset($result['status']) && $result['status'] == self::VERIFONE_ENV_CAPTURE_EVENT_TYPE_STATUS_AUTHORIZED) {
            $this->createTransactionForCapturePayment($order, $payment, $transactionId, $result);
        }
        if (isset($result['status']) && $result['status'] == self::VERIFONE_ENV_CAPTURE_EVENT_TYPE_STATUS_SETTLEMENT_COMPLETED) {
            $this->createTransactionForCapturePayment($order, $payment, $transactionId, $result);
        }
        if (isset($result['status']) && $result['status'] == self::VERIFONE_ENV_CAPTURE_EVENT_TYPE_STATUS_VOIDED) {
            $this->createTransactionForVoid($order, $payment, $transactionId, $result);
        }
    }

    /**
     * @return mixed
     */
    public function checkManualCapture()
    {
        return $this->_verifoneSettings->isManualCapture();
    }

    /**
     * @param $paymentMethod
     * @param $params
     *
     * @return mixed
     * @throws Exception
     */
    public function checkPaymentResponse($paymentMethod, $params)
    {
        if (! isset($params['checkout_id'])) {
            $this->_logger->critical('Error message', [
                'exception' => sprintf(
                    'The checkoutId parameter could not be found, in order to proceed it must exist. Current GET params: %s',
                    json_encode($params)
                )
            ]);

            return false;
        }

        if (isset($params['error_code'])) {
            $this->_logger->critical('Error message', [
                'exception' => sprintf(
                    'There was an error with the payment. Current GET params: %s',
                    json_encode($params)
                )
            ]);

            return false;
        }

        $this->setUpVerifoneSettings($paymentMethod);
        $this->getRegionMapper();
        $auth          = new BasicAuth($this->_verifoneSettings);
        $this->_client = new Client(
            $this->_mapper,
            $this->_verifoneSettings,
            $auth,
            $this->_simpleCurl
        );

        try {
            $response = $this->_client->getCheckout($params['checkout_id']);
        } catch (HttpException $exception) {
            $this->_logger->critical(sprintf(
                'There was an error in the api response, last known error was "%s", in file %s, at line %s. Error message: %s',
                json_last_error(),
                __FILE__,
                __LINE__,
                $exception->getMessage()
            ));

            return $this->failure();
        }

        if (! isset($response['events']) ||
             ( ! isset($response['merchant_reference'])) ||
             ( ! isset($response['transaction_id']))) {
            $this->_logger->critical('Error message', [
                'exception' => sprintf(
                    'The events or merchant_reference or transaction_id property could not be found in response, in order to proceed it must exist. API response %s',
                    json_encode($response)
                )
            ]);
            return false;
        }

        return $response;
    }

    /**
     * @param $paymentMethod
     * @param $quote
     *
     * @return array
     * @throws Exception
     */
    public function processPayment($paymentMethod, $quote)
    {
        $this->setUpVerifoneSettings($paymentMethod);
        $this->getRegionMapper();
        $auth          = new BasicAuth($this->_verifoneSettings);
        $this->_client = new Client(
            $this->_mapper,
            $this->_verifoneSettings,
            $auth,
            $this->_simpleCurl
        );

        if ($this->_verifoneSettings->is3dsEnabled() && null === $this->_verifoneSettings->get3dsContractId()) {
            $this->_logger->critical(
                'Error message',
                ['exception' => '3DS is enabled but no 3DS contract ID is provided in the configuration. Please fill in the field "3DS Contract Id" or disable the field "Enable/Disable 3d-secure" in the plugin settings.']
            );
            throw new Exception('An unexpected error occurred. Please retry the payment. No money has been charged.');
        }

        $billingAddress  = $quote->getBillingAddress();
        $shippingAddress = $quote->getShippingAddress();
        $billingArray    = $this->getBillingArray($billingAddress);
        $shippingArray   = $this->getShippingArray($shippingAddress, $billingAddress);

        $customerInfo = [
            'email_address' => substr($billingAddress->getEmail(), 0, 250),
            'entity_id'     => $this->_verifoneSettings->getEntityId()
        ];

        if (! empty($billingAddress->getTelephone())) {
            $customerInfo['phone_number'] = preg_replace('/[^0-9]+/', '', substr($billingAddress->getTelephone(), 0, 20));
        }

        if (! empty($billingAddress->getCompany())) {
            $customerInfo['company_name'] = substr($billingAddress->getCompany(), 0, 100);
        }

        $customerInfo = array_merge($customerInfo, $billingArray);
        $customerInfo = array_merge($customerInfo, $shippingArray);

        try {
            $customer = $this->_client->postCustomer($customerInfo);
        } catch (HttpException $exception) {
            $this->_logger->critical(sprintf(
                'There was an error in the api response, last known error was "%s", in file %s, at line %s. Error message: %s',
                json_last_error(),
                __FILE__,
                __LINE__,
                $exception->getMessage()
            ));

            return $this->failure();
        }
        $customerId = ( ! empty($customer['id'])) ? $customer['id'] : null;

        $currency      = $quote->getCurrency()->getQuoteCurrencyCode();
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $getLocale     = $objectManager->get('Magento\Framework\Locale\Resolver');
        $locale        = $getLocale->getLocale();
        $checkout      = $this->getCheckoutParams($quote, $paymentMethod, $currency, $customerId, $locale);

        try {
            $checkout = $this->_client->postCheckout($checkout);
        } catch (HttpException $exception) {
            $this->_logger->critical(sprintf(
                'There was an error in the api response, last known error was "%s", in file %s, at line %s. Error message: %s',
                json_last_error(),
                __FILE__,
                __LINE__,
                $exception->getMessage()
            ));

            return $this->failure();
        }
        $checkoutUrl = ( ! empty($checkout['url'])) ? $checkout['url'] : null;

        if ($this->_verifoneSettings->getPaymentType() === Settings::PAYMENT_TYPE_IFRAME) {
            $scheme = parse_url($checkout['url'], PHP_URL_SCHEME);
            $host   = parse_url($checkout['url'], PHP_URL_HOST);
            $url    = sprintf('%s://%s/v2/loader.js?checkoutId=%s', $scheme, $host, $checkout['id']);

            return [
                'method' => $this->_verifoneSettings->getPaymentType(),
                'url'    => $url
            ];
        }

        return [
            'method'   => $this->_verifoneSettings->getPaymentType(),
            'redirect' => $checkoutUrl,
        ];
    }

    /**
     * @throws \VerifoneEcomAPI\ApiWrapper\Http\HttpException
     * @throws Exception
     */
    public function processRefund($paymentMethod, $amount, $reason, $order, $payment)
    {
        $this->setUpVerifoneSettings($paymentMethod);
        $this->getRegionMapper();
        $auth                  = new BasicAuth($this->_verifoneSettings);
        $this->_client         = new Client(
            $this->_mapper,
            $this->_verifoneSettings,
            $auth,
            $this->_simpleCurl
        );
        $paymentAdditionalInfo = $payment->getAdditionalInformation(\Magento\Sales\Model\Order\Payment\Transaction::RAW_DETAILS);

        if (! isset($paymentAdditionalInfo['transactionId']) && ! isset($paymentAdditionalInfo['transaction_id'])) {
            throw new Exception('The order was not properly complete. The required parameter Transaction ID is missing from order information. Online refund is not possible. Please contact customer support.');
        }

        $transactionId  = ( ! empty($paymentAdditionalInfo['transactionId'])) ? $paymentAdditionalInfo['transactionId'] : $paymentAdditionalInfo['transaction_id'];
        $paymentProduct = $payment->getAdditionalInformation('verifone_payment_product') ?? null;

        if ($paymentProduct === 'paypal') {
            $refund = $this->getPayPalRefundParams($amount, $reason, $order->getOrderCurrencyCode());
        } else {
            $refund = $this->getRefundParams($amount, $reason, $order, $transactionId);
        }

        try {
            $response = $this->_client->postRefund($refund, $transactionId, $paymentProduct);
        } catch (HttpException $exception) {
            $this->_logger->critical(sprintf(
                'There was an error in the api response, last known error was "%s", in file %s, at line %s. Error message: %s',
                json_last_error(),
                __FILE__,
                __LINE__,
                $exception->getMessage()
            ));

            return $this->failure();
        }

        if (json_last_error() !== JSON_ERROR_NONE) {
            $this->_logger->critical('Error message', [
                'exception' => sprintf(
                    'There was an error decoding the json response from the api response, last known error was "%s", in file %s, at line %s. API response: %s',
                    json_last_error(),
                    __FILE__,
                    __LINE__,
                    $response
                )
            ]);
            throw new Exception('There was an error in the application');
        }
        if (! isset($response['id'])
             || ! isset($response['status'])
             || ! in_array($response['status'], $this->refundStatusOk)
        ) {
            $this->_logger->critical('Error message', ['exception' => ' Refund Error: Unable to refund transaction.']);
            throw new Exception('Refund Error: Unable to refund transaction.');
        }
        $payment->setAdditionalInformation(
            [\Magento\Sales\Model\Order\Payment\Transaction::RAW_DETAILS => (array) $response]
        );
        $payment->save();
        $order->setState(Order::STATE_CLOSED)
              ->addStatusToHistory(Order::STATE_CLOSED, __('Order payment is refunded with transaction ID %1', $response['id']), false)
              ->setStatus($order->getConfig()->getStateDefaultStatus(Order::STATE_CLOSED));
        $order->save();
    }

    /**
     * @param $paymentMethod
     * @param  string  $transactionId
     *
     * @return mixed
     * @throws Exception
     */
    public function getTransaction($paymentMethod, $transactionId)
    {
        $this->setUpVerifoneSettings($paymentMethod);
        $this->getRegionMapper();
        $auth          = new BasicAuth($this->_verifoneSettings);
        $this->_client = new Client(
            $this->_mapper,
            $this->_verifoneSettings,
            $auth,
            $this->_simpleCurl
        );

        try {
            $response = $this->_client->getTransaction($transactionId);
        } catch (HttpException $exception) {
            $this->_logger->critical(sprintf(
                'There was an error in the api response, last known error was "%s", in file %s, at line %s. Error message: %s',
                json_last_error(),
                __FILE__,
                __LINE__,
                $exception->getMessage()
            ));

            return $this->failure();
        }

        return $response;
    }

    /**
     * @param $paymentMethod
     * @param  string  $transactionId
     *
     * @return mixed
     * @throws Exception
     */
    public function getCheckout($paymentMethod, $checkoutId)
    {
        $this->setUpVerifoneSettings($paymentMethod);
        $this->getRegionMapper();
        $auth          = new BasicAuth($this->_verifoneSettings);
        $this->_client = new Client(
            $this->_mapper,
            $this->_verifoneSettings,
            $auth,
            $this->_simpleCurl
        );

        try {
            $response = $this->_client->getCheckout($checkoutId);
        } catch (HttpException $exception) {
            $this->_logger->critical(sprintf(
                'There was an error in the api response, last known error was "%s", in file %s, at line %s. Error message: %s',
                json_last_error(),
                __FILE__,
                __LINE__,
                $exception->getMessage()
            ));

            return $this->failure();
        }

        return $response;
    }

    /**
     * @param $amount
     * @param $reason
     * @param $currency
     *
     * @return array
     */
    public function getPayPalRefundParams($amount, $reason, $currency)
    {

        return [
            'amount' => [
                'value'        => $this->convertAmountToVerifone($amount, $currency),
                'currencyCode' => $currency,
            ],
            'reason' => trim($reason) != '' ? $reason : 'Refunded by merchant',
        ];
    }
    /**
     * @param $amount
     * @param $reason
     * @param $order
     * @param $transactionId
     *
     * @return array
     */
    public function getRefundParams($amount, $reason, $order, $transactionId)
    {
        $date = new \DateTime('now');

        return [
            'amount'            => $this->convertAmountToVerifone($amount, $order->getOrderCurrencyCode()),
            'reason'            => trim($reason) != '' ? $reason : 'Refunded by merchant',
            'id'                => "{$transactionId}",
            'reference_id'      => "{$order->getQuoteId()}",
            'created_date_time' => "{$date->format( 'Y-m-d\TH:i:s\Z' )}"
        ];
    }

    /**
     * @param $quote
     * @param $paymentMethod
     * @param $currency
     * @param $customerId
     * @param $locale
     *
     * @return array|\array[][]
     */
    public function getCheckoutParams($quote, $paymentMethod, $currency, $customerId, $locale)
    {
        $customer_data = [];
        if ($customerId !== '') {
            $customer_data['customer'] = $customerId;
        }

        $configurations = ['configurations' => []];

        $amount = $this->convertAmountToVerifone($quote->getGrandTotal(), $currency);

        if (in_array($locale, self::LOCALE_NL_CODES)) {
            $locale = 'nl-be';
        } elseif (in_array($locale, self::LOCALE_NO_CODES)) {
            $locale = 'no';
        } else {
            $locale = substr($locale, 0, 2);
        }

        $params = [
            'entity_id'          => $this->_verifoneSettings->getEntityId(),
            'currency_code'      => strtoupper($currency),
            'amount'             => $amount,
            'shop_url'           => $paymentMethod->getRedirectUrl(),
            'return_url'         => $paymentMethod->getReturnUrl().'?refNo='.$quote->getId(),
            'merchant_reference' => sprintf('%s', $quote->getId()),
            'i18n'               => [
                // using substr is the recommended way to get the language
                // also silently fallback to en if the language is not supported
                'default_language' => $locale,
            ],
        ];

        if ($this->_verifoneSettings->getIsCardEnabled()) {
            $configurations['configurations']['card'] = [
                'capture_now'         => ! $this->_verifoneSettings->isManualCapture(),
                'payment_contract_id' => $this->_verifoneSettings->getPaymentContractId(),
                'authorization_type'  => $this->_verifoneSettings->getAuthType(),
            ];

            if ($this->_verifoneSettings->is3dsEnabled() && $this->_verifoneSettings->get3dsContractId()) {
                $configurations['configurations']['card']['threed_secure'] = [
                    'threeds_contract_id' => $this->_verifoneSettings->get3dsContractId(),
                    'enabled'             => $this->_verifoneSettings->is3dsEnabled(),
                    'transaction_mode'    => $this->_verifoneSettings->getDeviceType(),
                ];
            }

            if ($this->_verifoneSettings->isFraudCheck()) {
                $configurations['configurations']['card']['fraud_protection_contract'] = $this->_verifoneSettings->getFraudCheckContractId();
            }
        }

        if ($this->_verifoneSettings->getThemeId()) {
            $configurations['theme_id'] = $this->_verifoneSettings->getThemeId();
        }

        if ($this->_verifoneSettings->getIsGooglePayEnabled()) {
            $configurations['configurations']['google_pay'] = [
                'card'        => [
                    'sca_compliance_level' => $this->_verifoneSettings->getGooglePayScaComplianceLevel(),

                    'payment_contract_id' => $this->_verifoneSettings->getGooglePayContractId(),
                ],
                'capture_now' => ! $this->checkManualCapture(),
            ];

            if (in_array($this->_verifoneSettings->getGooglePayScaComplianceLevel(), ['WALLET', 'FORCE_3DS'])) {
                $configurations['configurations']['google_pay']['card']['threed_secure'] = [
                    'threeds_contract_id' => $this->_verifoneSettings->getGooglePayThreedsContractId(),
                    'transaction_mode' => $this->getDevice(),
                ];
            }
        }

        if ($this->_verifoneSettings->getIsApplePayEnabled()) {
            $configurations['configurations']['apple_pay'] = [
                'card'        => [
                    'sca_compliance_level' => $this->_verifoneSettings->getApplePayScaComplianceLevel(),
                    'payment_contract_id'  => $this->_verifoneSettings->getApplePayContractId(),
                ],
                'capture_now' => ! $this->checkManualCapture(),
            ];

            if ('FORCE_3DS' === $this->_verifoneSettings->getApplePayScaComplianceLevel()) {
                $configurations['configurations']['apple_pay']['card']['threed_secure'] = [
                    'threeds_contract_id' => $this->_verifoneSettings->getApplePayThreedsContractId(),
                    'transaction_mode'    => $this->getDevice(),
                ];
            }
        }

        if ($this->_verifoneSettings->getVippsEnabled()) {
            $configurations['configurations']['vipps'] = [
                'capture_now'         => ! $this->checkManualCapture(),
                'payment_contract_id' => $this->_verifoneSettings->getVippsContractId(),
                'card'                => [
                    'sca_compliance_level' => $this->_verifoneSettings->getVippsScaLevel(),
                ]
            ];

            if (in_array($this->_verifoneSettings->getVippsScaLevel(), ['WALLET', 'FORCE_3DS'])) {
                $configurations['configurations']['vipps']['card']['threed_secure'] = [
                    'threeds_contract_id' => $this->_verifoneSettings->getVipps3DSContractId(),
                    'transaction_mode' => $this->getDevice(),
                ];
            }
        }

        if ($this->_verifoneSettings->getMobilePayEnabled()) {
            $configurations['configurations']['mobile_pay'] = [
                'capture_now'         => ! $this->checkManualCapture(),
                'payment_contract_id' => $this->_verifoneSettings->getMobilePayContractId(),
                'card'                => [
                    'sca_compliance_level' => $this->_verifoneSettings->getMobilePayScaLevel(),
                ]
            ];

            if (in_array($this->_verifoneSettings->getMobilePayScaLevel(), ['WALLET', 'FORCE_3DS'])) {
                $configurations['configurations']['mobile_pay']['card']['threed_secure'] = [
                    'threeds_contract_id' => $this->_verifoneSettings->getMobilePay3DSContractId(),
                    'transaction_mode'    => $this->getDevice(),
                    'enabled'    => true,
                ];
            }
        }

        if ($this->_verifoneSettings->getIsPaypalEnabled()) {
            $configurations['configurations']['paypal'] = [
                'capture_now'         => ! $this->checkManualCapture(),
                'payment_contract_id' => $this->_verifoneSettings->getPaypalContractId(),
            ];
        }

        if ($this->_verifoneSettings->getGpp2Enabled()) {
            $configurations['configurations']['gpp2'] = [
                'payment_contract_id' => $this->_verifoneSettings->getGpp2ContractId(),
            ];
        }
        if ($this->_verifoneSettings->getIsSwishEnabled()) {
            $configurations['configurations']['swish'] = [
                'shopper_interaction' => 'ECOMMERCE',
            ];
        }
        if ($this->_verifoneSettings->getIsKlarnaEnabled()) {
            $configurations['configurations']['klarna'] = [
                'capture_now' => ! $this->checkManualCapture(),
            ];
        }

        if ($this->_verifoneSettings->getAbsInstallmentsEnabled() && $this->_verifoneSettings->getIsCardEnabled()) {
            $installment = $this->_verifoneSettings->getAbsInstallmentsForAmount($quote->getGrandTotal());

            if ($installment) {
                $configurations['configurations']['card']['credit_term'] = 'INSTALMENT_STANDARD';
                $configurations['configurations']['card']['instalment']  = [
                    'max_number_of_instalments' => $installment
                ];
            } else {
                $configurations['configurations']['card']['credit_term'] = 'STANDARD';
            }
        }

        if ($this->_verifoneSettings->getIsInvoice4uEnabled() && $this->_verifoneSettings->getIsCardEnabled()) {
            $configurations['receipt_type'] = $this->_verifoneSettings->getInvoice4uReceiptType();
        }

        if (($this->_verifoneSettings->getIsInvoice4uEnabled() && $this->_verifoneSettings->getIsCardEnabled()) || $this->_verifoneSettings->getIsKlarnaEnabled()) {
            $itemsCollection = $quote->getItemsCollection();
            $items = $itemsCollection->getItems();

            foreach ($items as $item) {
                $unitPriceExclTax=$this->convertAmountToVerifone($item->getBasePrice(), $currency);
                $unitPriceInclTax = $this->convertAmountToVerifone($item->getBasePriceInclTax(), $currency);
                $taxPercent=(int)$item->getTaxPercent();
                $priceTotal =round($unitPriceExclTax*(1+($taxPercent/100)) * $item->getQty());

                $lineItem = [
                    'name' => substr(preg_replace('/[^A-Za-z0-9\-]/', ' ', $item->getName()), 0, 248),
                    'quantity' => $item->getQty(),
                    'unit_price' => round($unitPriceExclTax),
                    'total_amount' => round($priceTotal),
                ];

                if ($this->_verifoneSettings->getInvoice4uSendTaxRates() || $this->_verifoneSettings->getIsKlarnaEnabled()) {
                    $lineItem['tax_rate'] = $taxPercent;
                    $lineItem['total_tax_amount']=round(($unitPriceExclTax*$item->getQty())*($taxPercent/100));
                }

                $configurations['line_items'][] = $lineItem;
            }

            $address = $quote->getShippingAddress();
            $address->collectShippingRates();
            $shippingData=$address->getData();
            $shippingAmount=intval((float)$shippingData['shipping_amount']*100);
            if ($shippingAmount>0) {
                $lineItem=[
                    'name'=>'Shipping',
                    'quantity'=>1,
                    'unit_price'=>$shippingAmount,
                    'total_amount'=>$shippingAmount,
                ];

                if ($this->_verifoneSettings->getInvoice4uSendTaxRates() || $this->_verifoneSettings->getIsKlarnaEnabled()) {
                    $lineItem['tax_rate'] = 0;
                    $lineItem['total_tax_amount']=0;
                }

                $configurations['line_items'][]=$lineItem;
            }
        }

        $checkoutData = array_merge($params, $customer_data);
        return array_merge($checkoutData, $configurations);
    }

    /**
     * @param $billingAddress
     *
     * @return array
     */
    public function getBillingArray($billingAddress)
    {
        $billingArr = [];

        if (! empty($billingAddress->getEmail())) {
            $billingArr['email_address'] = substr($billingAddress->getEmail(), 0, 250);
        }

        if (! empty($billingAddress->getTelephone())) {
            $billingArr['phone_number'] = preg_replace('/[^0-9]+/', '', substr($billingAddress->getTelephone(), 0, 20));
        }

        if (! empty($billingAddress->getCompany())) {
            $billingArr['company_name'] = substr($billingAddress->getCompany(), 0, 100);
        }

        if (! empty($billingAddress->getStreet()[0])) {
            $billingArr['billing']['address_1'] = substr($billingAddress->getStreet()[0], 0, 40);
        }

        if (! empty($billingAddress->getStreet()[1])) {
            $billingArr['billing']['address_2'] = substr($billingAddress->getStreet()[1], 0, 40);
        }

        if (! empty($billingAddress->getCity())) {
            $billingArr['billing']['city'] = substr($billingAddress->getCity(), 0, 40);
        }

        if (! empty($billingAddress->getCountryId())) {
            $billingArr['billing']['country_code'] = strtoupper($billingAddress->getCountryId());
        }

        if (! empty($billingAddress->getFirstname())) {
            $billingArr['billing']['first_name'] = substr($billingAddress->getFirstname(), 0, 50);
        }

        if (! empty($billingAddress->getLastname())) {
            $billingArr['billing']['last_name'] = substr($billingAddress->getLastname(), 0, 50);
        }

        if (! empty($billingAddress->getTelephone())) {
            $billingArr['billing']['phone'] = preg_replace('/[^0-9]+/', '', substr($billingAddress->getTelephone(), 0, 20));
        }

        if (! empty($billingAddress->getPostcode())) {
            $billingArr['billing']['postal_code'] = substr($billingAddress->getPostcode(), 0, 10);
        }

        if (! empty($billingAddress)) {
            $billingArr['billing']['state'] = ( ! empty($billingAddress->getRegion())) ? substr($billingAddress->getRegion(), 0, 10) : 'XX';
        }

        return $billingArr;
    }

    /**
     * @param $shippingAddress
     * @param $billingAddress
     *
     * @return array
     */
    public function getShippingArray($shippingAddress, $billingAddress)
    {
        $shippingArr = [];

        if (! empty($shippingAddress->getStreet()[0])) {
            $shippingArr['shipping']['address_1'] = substr($shippingAddress->getStreet()[0], 0, 40);
        } elseif (! empty($billingAddress->getStreet()[0])) {
            $shippingArr['shipping']['address_1'] = substr($billingAddress->getStreet()[0], 0, 40);
        }

        if (! empty($shippingAddress->getStreet()[1])) {
            $shippingArr['shipping']['address_2'] = substr($shippingAddress->getStreet()[1], 0, 40);
        } elseif (! empty($billingAddress->getStreet()[1])) {
            $shippingArr['shipping']['address_2'] = substr($billingAddress->getStreet()[1], 0, 40);
        }

        if (! empty($shippingAddress->getCity())) {
            $shippingArr['shipping']['city'] = substr($shippingAddress->getCity(), 0, 40);
        } elseif (! empty($billingAddress->getCity())) {
            $shippingArr['shipping']['city'] = substr($billingAddress->getCity(), 0, 40);
        }

        if (! empty($shippingAddress->getCountryId())) {
            $shippingArr['shipping']['country_code'] = strtoupper($shippingAddress->getCountryId());
        } elseif (! empty($billingAddress->getCountryId())) {
            $shippingArr['shipping']['country_code'] = strtoupper($billingAddress->getCountryId());
        }

        if (! empty($shippingAddress->getFirstname())) {
            $shippingArr['shipping']['first_name'] = substr($shippingAddress->getFirstname(), 0, 50);
        } elseif (! empty($billingAddress->getFirstname())) {
            $shippingArr['shipping']['first_name'] = substr($billingAddress->getFirstname(), 0, 50);
        }

        if (! empty($shippingAddress->getLastname())) {
            $shippingArr['shipping']['last_name'] = substr($shippingAddress->getLastname(), 0, 50);
        } elseif (! empty($billingAddress->getLastname())) {
            $shippingArr['shipping']['last_name'] = substr($billingAddress->getLastname(), 0, 50);
        }

        if (! empty($shippingAddress->getTelephone())) {
            $shippingArr['shipping']['phone'] = preg_replace('/[^0-9]+/', '', substr($shippingAddress->getTelephone(), 0, 20));
        } elseif (! empty($billingAddress->getTelephone())) {
            $shippingArr['shipping']['phone'] = preg_replace('/[^0-9]+/', '', substr($billingAddress->getTelephone(), 0, 20));
        }

        if (! empty($shippingAddress->getPostcode())) {
            $shippingArr['shipping']['postal_code'] = substr($shippingAddress->getPostcode(), 0, 10);
        } elseif (! empty($billingAddress->getPostcode())) {
            $shippingArr['shipping']['postal_code'] = substr($billingAddress->getPostcode(), 0, 10);
        }

        if (! empty($shippingAddress)) {
            $shippingArr['shipping']['state'] = ( ! empty($shippingAddress->getRegion())) ? substr($shippingAddress->getRegion(), 0, 10) : 'XX';
        }

        return $shippingArr;
    }

    /**
     * @return array
     */
    private function failure()
    {
        return [
            'result'   => 'failure',
            'messages' => 'There has been an error processing your order',
        ];
    }

    /**
     * @return \Magento\Sales\Model\Service\InvoiceService
     */
    public function getInvoiceService()
    {
        return $this->_invoiceService;
    }

    /**
     * @return \Magento\Framework\DB\TransactionFactory
     */
    public function getTransactionFactory()
    {
        return $this->_transactionFactory;
    }

    /**
     * @param $order
     * @param $payment
     * @param $transactionId
     * @param $result
     */
    public function createTransactionForCapturePayment($order, $payment, $transactionId, $result)
    {
        $message     = sprintf('Order with transaction ID %s was authorized. Authorization ID %s', $transactionId, $result['id']);
        $transaction = $this->_transactionBuilder->setPayment($payment)
                                                 ->setOrder($order)
                                                 ->setTransactionId($transactionId)
                                                 ->setFailSafe(true)
                                                 ->build(\Magento\Sales\Model\Order\Payment\Transaction::TYPE_CAPTURE);
        $payment->addTransactionCommentsToOrder($transaction, $message);
        $payment->save();
    }

    /**
     * @param $order
     * @param $payment
     * @param $transactionId
     * @param $result
     */
    public function createTransactionForVoid($order, $payment, $transactionId, $result)
    {
        $message     = sprintf('Order with transaction ID %s was voided. Void ID %s', $transactionId, $result['id']);
        $trans       = $this->_transactionBuilder;
        $transaction = $trans->setPayment($payment)
                             ->setOrder($order)
                             ->setTransactionId($transactionId)
                             ->setFailSafe(true)
                             ->build(\Magento\Sales\Model\Order\Payment\Transaction::TYPE_VOID);

        $payment->addTransactionCommentsToOrder($transaction, $message);
        $payment->save();
    }

    public function checkVerifoneTransactionIdInOrderLogHistory(string $verifoneTransactionId, Order $order): bool
    {
        $statusHistory = $order->getStatusHistoryCollection()->toArray();

        foreach ($statusHistory['items'] as $statusHistoryItem) {
            if (!empty($statusHistoryItem['comment']) && strpos($statusHistoryItem['comment'], $verifoneTransactionId) !== false) {
                return true;
            }
        }

        return false;
    }

    public function convertAmountToVerifone($amount, $currency): float
    {
        if (in_array(strtolower($currency), self::ZERO_DECIMAL_CURRENCIES)) {
            $verifoneAmount = round($amount);
        } else {
            $verifoneAmount = round($amount * 100);
        }

        return $verifoneAmount;
    }

    public function convertAmountToMagento($amount, $currency): float
    {
        if (in_array(strtolower($currency), self::ZERO_DECIMAL_CURRENCIES)) {
            $magentoAmount = (floatval($amount));
        } else {
            $magentoAmount = (floatval($amount) / 100);
        }

        return $magentoAmount;
    }

    /**
     * Mobile ore Computer
     * @return false|int
     */
    private function getDevice()
    {
        return preg_match(
            "/(android|avantgo|blackberry|bolt|boost|cricket|docomo|fone|hiptop|mini|mobi|palm|phone|pie|tablet|up\.browser|up\.link|webos|wos)/i",
            $_SERVER["HTTP_USER_AGENT"]
        ) ? 'P' : 'S';
    }
}
