<?php

namespace Verifone\Hosted\Model;

use Exception;
use Magento\Sales\Model\Order;

class Verifone extends \Magento\Payment\Model\Method\AbstractMethod
{

    const CODE = 'verifone_hosted';

    protected $_code = self::CODE;

    protected $_isGateway = false;

    protected $_isOffline = false;

    protected $_canRefund = true;

    protected $_canRefundInvoicePartial = true;

    protected $_canOrder = true;

    protected $_canCapture = true;

    protected $_canAuthorize = true;

    protected $_canCapturePartial = false;

    protected $_canCaptureOnce = false;

    protected $_canVoid = true;

    protected $_isInitializeNeeded = false;

    protected $_helper;

    protected $_formBlockType = \Verifone\Hosted\Block\Form\Verifone::class;

    protected $_infoBlockType = \Verifone\Hosted\Block\Info\Verifone::class;

    protected $_orderSender;

    protected $_orderRepo;

    protected $_transactionBuilder;

    protected $_taxHelper;

    protected $_locale_resolver;

    protected $_store;

    protected $_request;


    /**
     * Verifone constructor.
     *
     * @param \Magento\Framework\Model\Context $context
     * @param \Magento\Framework\Registry $registry
     * @param \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory
     * @param \Magento\Framework\Api\AttributeValueFactory $customAttributeFactory
     * @param \Magento\Payment\Helper\Data $paymentData
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param \Magento\Payment\Model\Method\Logger $logger
     * @param \Verifone\Hosted\Helper\Verifone $helper
     * @param \Magento\Sales\Model\Order\Email\Sender\OrderSender $orderSender
     * @param \Magento\Sales\Api\OrderRepositoryInterface $orderRepo
     * @param \Magento\Sales\Model\Order\Payment\Transaction\BuilderInterface $transactionBuilder
     * @param \Magento\Catalog\Helper\Data $taxHelper
     * @param \Magento\Framework\Locale\Resolver $locale_resolver
     * @param \Magento\Store\Api\Data\StoreInterface $store
     */
    public function __construct(
        \Magento\Framework\Model\Context $context,
        \Magento\Framework\Registry $registry,
        \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory,
        \Magento\Framework\Api\AttributeValueFactory $customAttributeFactory,
        \Magento\Payment\Helper\Data $paymentData,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Payment\Model\Method\Logger $logger,
        \Verifone\Hosted\Helper\Verifone $helper,
        \Magento\Sales\Model\Order\Email\Sender\OrderSender $orderSender,
        \Magento\Sales\Api\OrderRepositoryInterface $orderRepo,
        \Magento\Sales\Model\Order\Payment\Transaction\BuilderInterface $transactionBuilder,
        \Magento\Catalog\Helper\Data $taxHelper,
        \Magento\Framework\Locale\Resolver $locale_resolver,
        \Magento\Store\Api\Data\StoreInterface $store,
        \Magento\Framework\App\RequestInterface $request
    ) {
        $this->_helper             = $helper;
        $this->_orderSender        = $orderSender;
        $this->_orderRepo          = $orderRepo;
        $this->_transactionBuilder = $transactionBuilder;
        $this->_taxHelper          = $taxHelper;
        $this->_locale_resolver    = $locale_resolver;
        $this->_store              = $store;
        $this->_request            = $request;

        parent::__construct(
            $context,
            $registry,
            $extensionFactory,
            $customAttributeFactory,
            $paymentData,
            $scopeConfig,
            $logger
        );
    }

    /**
     * @param $response
     */
    protected function _createTransaction($response, $transactionFactory, $order, $invoice = null)
    {
        $transaction = null;
        if ($invoice) {
            $transaction = $transactionFactory->create()
                ->addObject($invoice)
                ->addObject($order);

            $transaction->save();
        }
        $order->save();
    }

    /**
     * @param Order  $order
     * @param $response
     * @param $manualCapture
     * @param string  $paymentProduct
     * @param int   $amountPaid
     *
     * @return Order
     * @throws Exception
     */
    public function postProcessing(Order $order, $response, $manualCapture, ?string $paymentProduct, int $amountPaid): Order
    {
        if (null === $paymentProduct) {
            $paymentProduct = 'unknown';
        }
        
        try {
            $transactionId = $response['transaction_id'];
            if ($manualCapture) {
                $this->manualCapturePaymentSetup($order, $transactionId, $response, $paymentProduct);
                $holdStatus = Order::STATE_HOLDED;
                $order->setStatus($holdStatus);
                $order->save();
                $this->_orderRepo->save($order);
                return $order;
            }

            $payment = $order->getPayment();
            $payment->setTransactionId($transactionId);
            $payment->setIsTransactionClosed(0);
            $payment->setAdditionalInformation(
                [\Magento\Sales\Model\Order\Payment\Transaction::RAW_DETAILS => (array)$response,
                    'verifone_order_status' => 'approved',
                    'verifone_order_number' => $transactionId,
                    'verifone_payment_product' => $paymentProduct
                ]
            );
            $payment->setShouldCloseParentTransaction(true);
            $payment->setParentTransactionId(null);
            $payment->save();
            //Only create the transaction here & only if no invoice creation trigger is set in Magento2 admin, connectors settings
            $formattedPrice = $order->getBaseCurrency()->formatTxt(
                $this->_helper->convertAmountToMagento($amountPaid, $order->getOrderCurrencyCode())
            );

            $message = __('The authorized amount is %1.', $formattedPrice);
            $trans = $this->_transactionBuilder;
            $transaction = $trans->setPayment($payment)
                ->setOrder($order)
                ->setTransactionId($transactionId)
                ->setFailSafe(true)
                ->build(\Magento\Sales\Model\Order\Payment\Transaction::TYPE_CAPTURE);

            $payment->addTransactionCommentsToOrder($transaction, $message);
            $orderAmount = $this->_helper->convertAmountToVerifone($order->getGrandTotal(), $order->getOrderCurrencyCode());

            if ($amountPaid >= $orderAmount) {
                $order->setStatus($this->_helper->getVerifoneSettings()->getOrderStatus())
                    ->setState($this->_helper->getVerifoneSettings()->getOrderState());
            } else {
                $payment->addTransactionCommentsToOrder($transaction, __('The transaction was not completed because the amount paid differs from the payment amount. Order amount is %1 , but was paid %2.', $orderAmount, $amountPaid));
            }

            $payment->save();
            $order->save();

            $this->_orderRepo->save($order);

            return $order;
        } catch (Exception $e) {
            throw new Exception(sprintf('Error Creating Invoice: "%s"', $e->getMessage()));
        }
    }

    /**
     * @return string
     */
    public function getRedirectUrl()
    {
        return $this->_helper->getUrl($this->getConfigData('redirect_url'));
    }

    /**
     * @return string
     */
    public function getReturnUrl()
    {
        return $this->_helper->getUrl($this->getConfigData('return_url'));
    }

    /**
     * @return string
     */
    public function getCancelUrl()
    {
        return $this->_helper->getUrl($this->getConfigData('cancel_url'));
    }

    /**
     * @return mixed
     */
    public function getOrderStatus()
    {
        return $this->getConfigData('order_status');
    }

    /**
     * @return mixed
     */
    public function getOrderState()
    {
        $status= $this->getConfigData('order_status');

        switch ($status) {
            case 'pending':
                return 'new';

            default:
                return $status;
        }
    }

    public function refund(\Magento\Payment\Model\InfoInterface $payment, $amount)
    {

        if ($amount <= 0) {
            throw new \Magento\Framework\Exception\LocalizedException(__('Invalid amount for refund.'));
        }

        if (! $this->_canRefund) {
            throw new \Magento\Framework\Exception\LocalizedException(__('The refund action is not available.'));
        }

        if (! $payment->getParentTransactionId()) {
            throw new \Magento\Framework\Exception\LocalizedException(__('Invalid transaction ID.'));
        }

        $data = $this->_request->getPost('creditmemo');
        $reason        = (!empty($data['comment_text'])) ? $data['comment_text'] : '';
        $order         = $payment->getOrder();
        $paymentMethod = $payment->getMethodInstance();

        $this->_helper->processRefund($paymentMethod, $amount, $reason, $order, $payment);

        return $this;
    }

    private function manualCapturePaymentSetup($order, $transactionId, $response, $paymentProduct)
    {
        $payment = $order->getPayment();
        $payment->setTransactionId($transactionId);
        $payment->setAdditionalInformation('manual_capture', 'yes');
        $payment->setAdditionalInformation(
            'verifone_order_number',
            $transactionId
        );
        $payment->setAdditionalInformation(
            'verifone_payment_product',
            $paymentProduct
        );
        $payment->setAdditionalInformation(
            [ \Magento\Sales\Model\Order\Payment\Transaction::RAW_DETAILS => (array) $response,
                'manual_capture' => 'yes',
                'verifone_order_number' => $transactionId,
                'verifone_payment_product' => $paymentProduct
            ]
        );
        // Formatted price
        $formatedPrice = $order->getBaseCurrency()->formatTxt($order->getGrandTotal());
        // create transaction
        $transaction = $this->manualCaptureTransactionSetup($order, $payment, $transactionId, $response);

        // Add transaction to payment
        $payment->addTransactionCommentsToOrder($transaction, __('The authorized amount is %1.', $formatedPrice));
        $payment->setParentTransactionId(null);
        $payment->save();
        $transaction->save();
    }

    private function manualCaptureTransactionSetup($order, $payment, $transactionId, $response)
    {
        $transaction = $this->_transactionBuilder->setPayment($payment)
            ->setOrder($order)
            ->setTransactionId($transactionId)
            ->setAdditionalInformation([ \Magento\Sales\Model\Order\Payment\Transaction::RAW_DETAILS => (array) $response ])
            ->setFailSafe(true)
            ->build(\Magento\Sales\Model\Order\Payment\Transaction::TYPE_CAPTURE);
        return $transaction;
    }
}
